
import re
import time
import os
from os.path import getmtime
from misc import debug, stripColor, warning
import events

class ParseError(Exception): pass
class EmptyFile(ParseError): pass
class FlagColorError(ParseError): pass

NAME_MAXLEN = 29

# server messages that dont have endline mark
SERVER_MSGS_WO_EOL = ("Match starting...",
                      'server:')

# osp rounds begin with FIGHT! message
# osp rounds end with stats printed out
OSP_ROUND_BEGIN_MSG = "^1FIGHT!\n"
OSP_MATCH_UNPAUSED = "^3Match is ^5UNPAUSED^3"
OSP_MATCH_REF_UNPAUSED = "^3Referee ^5UNPAUSES^3 the match ... resuming in 10 seconds!"

# when first osp round is printed, its stats are discarded, because
# they are summed to 2nd round.
OSP_CLOCK_SET = ">>> ^3Clock set to: "

OSP_STATS_BEGIN = "^7TEAM   Player          Kll " \
                  "Dth Sui TK Eff  ^3GP^7    ^2DG    ^1DR   ^6TD  ^3Score"
ETPRO_STATS_BEGIN = "^7TEAM   Player          Kll Dth Sui TK Eff ^3Gib^7"\
                  "    ^2DG    ^1DR   ^6TD  ^3Score"

OSP_OBJ_FAILED = ">>> ^3Objective NOT reached in time "
OSP_OBJ_SUCCES = ">>> ^3Objective reached at "

OSP_INTERMEDIATE_STAT = "Current time: ^3"

#OSP_STAT_ROUND_INFO = "^7Overall stats for: "

KILL_NOT_ALLOWED = "^3kill^7 not allowed during intermission."

class Q3log_reg_exps:
    def __init__(self):
        self.rename_re = re.compile(events.rename)
        self.mm1_re = re.compile(events.mm1)
        self.mm2_re = re.compile(events.mm2)
        s = r"(?P<name1>.+?)\^7 %s.$"
        
        self.suicides_res = [re.compile(s % e[1]) for e in events.suicides]
        self.kills_res = {}

        s = r"(?P<name1>.+?)\^7 %s (?P<name2>.+?)\^7%s$"
        for e in events.kills:
            self.kills_res[e[0]] = re.compile(s % e[1:])

        self.unknownkill_re = re.compile(events.unknownkill)        
        self.teamkill_re = re.compile(events.teamkill)
        self.etpro_tk_re = re.compile(r"^\^1TEAM KILL\^7: ")
        
        
class Q3Event:
    """Generic event in logfile"""
    ID = -1
    def __init__(self):
        Q3Event.ID += 1
        self.timestamp = Q3Event.ID

def mangle(name):
    return name[:29]

class Frag(Q3Event):
    "Attacked kills his Victim with Weapon"
    def __init__(self, attacker, victim, weapon):
        Q3Event.__init__(self)
        assert attacker != ""
        assert victim != ""
        self.attacker = mangle(attacker)
        self.victim = mangle(victim)
        self.weapon = weapon

class Teamkill(Q3Event):
    "Teamkill.. how rude"
    def __init__(self, attacker, victim):
        Q3Event.__init__(self)
        assert attacker != ""
        assert victim != ""
        self.attacker = mangle(attacker)
        self.victim = mangle(victim)

class Suicide(Q3Event):
    "Depressed person suicided"
    def __init__(self, depressed):
        Q3Event.__init__(self)
        assert depressed != ""
        self.depressed = mangle(depressed)

class Rename(Q3Event):
    "Player renamed"
    def __init__(self, before, after):
        Q3Event.__init__(self)
        assert before != ""
        assert after != ""
        self.before = mangle(before)
        self.after = mangle(after)

class Messagemode1(Q3Event):
    "Messagemode1 message (public talk)"
    def __init__(self, talker, msg):
        Q3Event.__init__(self)
        assert talker != ""
        self.talker = mangle(talker)
        self.msg = msg

class LogCreated(Q3Event):
    "First line of logfile is timestamp when logfile is created"
    def __init__(self, timestamp):
        Q3Event.__init__(self)
        self.timestamp = timestamp

class OspRoundStart(Q3Event):
    def __init__(self):
        Q3Event.__init__(self)

class OspMatchUnpaused(Q3Event):
    def __init__(self):
        Q3Event.__init__(self)

class OspRoundEnd(Q3Event):
    def __init__(self, sw):
        Q3Event.__init__(self)
        self.sw = sw

class OspPlayerStat(Q3Event):
    "osp stat info"
    def __init__(self, player, round, kill, death, suicide, teamkill, effiency,
                 gp, dg, dr, td, score):
        Q3Event.__init__(self)
        assert player != ""
        self.player = player
        self.kill = kill
        self.death = death
        self.suicide = suicide
        self.teamkill = teamkill
        self.effiency = effiency
        self.gp = gp
        self.dg = dg
        self.dr = dr
        self.td = td
        self.score = score
        self.round = round
        #print player, kill


def removeSkipNotify(line):
    if line.startswith('[skipnotify]'):
        line = line[len('[skipnotify]'):]
    return line

class Q3LogParser:
    """Parser logfile to series of events, which are saved to
    event_list(earlies first). """
    def __init__(self, qconsole_log_file):
        self.is_etpro = False

        # Load regular expressions
        self.reg_exps = Q3log_reg_exps()

        # parse events to this list
        self.event_list = self.parse_events(qconsole_log_file)

        
    def parse_events(self, fileName):        
        events = []

        file = open(fileName, "r")
        
        # start reading logfile, first line is timestamp
        line = file.readline()
        if not line:
            raise EmptyFile, "File %s is empty" 

        if line.find('logfile opened on ') == -1:
            warning("logfile did not start with timestamp, "
                    "but I am gonna try to parse it anyway")
            # timestamp it with creation date
            ts = time.asctime(time.localtime(os.stat(fileName).st_ctime))
            events.append(LogCreated(ts))
        else:
            # append timestamp 
            events.append(LogCreated(line[18:-1]))
        
        # dummy lcass used to jump out of from inner For-loops to out For-loop
        class matched(Exception): pass

        line = ""
        lines = file.readlines()
        #osp_stat_round = 0

        #osp_round_is_ending = 0
        
        while 1:
            if not lines:
                break
            
            prevLine = line            
            line = lines.pop(0)

            # args this code does not work in all situations
            # like "server: lollloollll<then some player message>"
            # or does it?
            for msg in SERVER_MSGS_WO_EOL:
                if line.startswith(msg):
                    line = line[len(msg):]
                    break

            line = removeSkipNotify(line)

            if line.startswith(KILL_NOT_ALLOWED):
                line = line[len(KILL_NOT_ALLOWED):]
                
            try:
                
                # OSP support
                if line == OSP_ROUND_BEGIN_MSG:
                    events.append(OspRoundStart())
                    raise matched

                if (line.startswith(OSP_MATCH_UNPAUSED) or
                    line.startswith(OSP_MATCH_REF_UNPAUSED)):
                    events.append(OspMatchUnpaused())
                    raise matched

                if line.startswith(OSP_INTERMEDIATE_STAT):
                    # read and ignore
                    tmp = []
                    self.readOSPStats(lines, tmp)
                    raise matched
                
                if line.startswith(OSP_STATS_BEGIN):
                    #if not osp_round_is_ending:
                    #    raise matched
                    self.readOSPStats(lines, events)
                    #osp_round_is_ending = 0
                    raise matched

                if line.startswith(ETPRO_STATS_BEGIN):
                    #if not osp_round_is_ending:
                    #    raise matched
                    self.is_etpro = True
                    self.readOSPStats(lines, events)
                    #osp_round_is_ending = 0
                    raise matched

#                 if line.startswith(OSP_STAT_ROUND_INFO):
#                     if line.endswith("(^20^7 Rounds)\n"):
#                         osp_stat_round = 0
#                     elif line.endswith("(^21^7 Round)\n"):
#                         osp_stat_round = 1
#                     elif line.endswith("(^22^7 Rounds)\n"):
#                         osp_stat_round = 2
#                     else:
#                         warning("unknown osp_stat_round_info line")
#                         warning(repr(line))
#                         osp_stat_round = -1
                        
#                     osp_round_is_ending = 1
#                     raise matched
                
                # mm2 ignore
                m = self.reg_exps.mm2_re.match(line)
                if m:
                    raise matched
                
                m = self.reg_exps.mm1_re.match(line)
                if m:
                    talker = m.group('talker')[:NAME_MAXLEN]
                    msg = m.group('line')
                    events.append(Messagemode1(talker, msg))
                    raise matched

                # etpro teamkills look bit different
                m = self.reg_exps.etpro_tk_re.match(line)
                if m:
                    tk_line = line[15:]
                    for weapon, kill_re in self.reg_exps.kills_res.items():
                        m = kill_re.match(tk_line)                    
                        if m:
                            killer = m.group('name2')
                            victim = m.group('name1')
                            events.append(Teamkill(killer, victim))
                            break
                    else:
                        warning("TK found, but could not parse further. "
                                "Report this bug to author.")
                    raise matched

                
                # check if this line match to any kill_msg
                for weapon, kill_re in self.reg_exps.kills_res.items():
                    m = kill_re.match(line)                    
                    if m:
                        killer = m.group('name2')
                        victim = m.group('name1')
                        events.append(Frag(killer, victim, weapon))
                        raise matched

                m = self.reg_exps.unknownkill_re.match(line)
                if m:
                    killer = m.group('name2')
                    victim = m.group('name1')
                    events.append(Frag(killer, victim, "Unknown"))
                    raise matched
                
                # suicides
                for suicide_re in self.reg_exps.suicides_res:                
                    m = suicide_re.match(line)                
                    if m:
                        suicider = m.group('name1')
                        events.append(Suicide(suicider))
                        raise matched

                # teamkill
                m = self.reg_exps.teamkill_re.match(line)
                if m:
                    killer = m.group('name2')
                    victim = m.group('name1')
                    events.append(Teamkill(killer, victim))
                    raise matched

                # rename
                m = self.reg_exps.rename_re.match(line)
                if m:                    
                    before = m.group('name1')[:NAME_MAXLEN]
                    after = m.group('name2')[:NAME_MAXLEN]
                    events.append(Rename(before, after))
                    raise matched
                
            except matched:
                continue
                

        # for line
        file.close()
        return events

    def readOSPStats(self, lines, events):
        sep = "^7" + "-"*69
        teamsProcessed = 0
        ospEvents = []
        while 1:
            if teamsProcessed == 2:
                break

            if not lines:
                warning("osp stats didn't have both teams data")
                events.append(OspRoundEnd(sw=0))
                return
            
            line = readLine(lines)            
            if line != sep:                
                continue

            # player stats follow
            while 1:
                line = readLine(lines)
                if line == sep:
                    teamsProcessed += 1
                    break
                
                # read in player line
                if not (line.startswith("^1Axis^7") or
                        line.startswith("^4Allies^7")):
                    raise ParseError("osp stat read error, team is not axis "
                                     "nor allies(%s)" % line)
                
#^4Allies^7 ^7Primalco.GS    ^3   3   5   0  0^7  37^3  13^2   608^1  1051^6    0^3     16                
                player = line[13:28].strip()
                rest = stripColor(line[28:], space=1)
                fields = rest.split()
                fields = map(int, fields)
                debug(fields)
                # sometimes osp bugs and print too big teamdamage value
                if fields[8] >= 10000:
                    warning("enemy territory has printed malformat stats, "
                            "therefore player %s teamdamage is not counted "
                            "for this round" % player)
                    fields[8] = 0
                ospEvents.append(OspPlayerStat(player, None,
                                               *fields))

        # seek if this was first sw round. if it was, discard
        # stats

        # when did this occur? maybe when not sw round
        if len(lines) < 4:
            events.extend(ospEvents)
            events.append(OspRoundEnd(sw=0))
            return

        line = lines[3]
        line = removeSkipNotify(line)
        if line == "\n":
            line = lines[6]
            line = removeSkipNotify(line)

        line = removeSkipNotify(line)
        
        sw = 0
        if (line.startswith(OSP_CLOCK_SET)):
            sw = 1
        elif (line.startswith(OSP_OBJ_SUCCES) or
              line.startswith(OSP_OBJ_FAILED)):
            sw = 2
        else:
            sw = 0

        # if sw = 0, then i guess player did /statdump ???
        #if sw == 0:
        #    return
        
        for e in ospEvents:
            e.round = sw
            
        events.extend(ospEvents)
        events.append(OspRoundEnd(sw))
        return

def readLine(lines):
    if not lines:
        raise ParseError("premature eof when reading osp stats!")

    line = lines.pop(0)                
    line = line.strip()
    return removeSkipNotify(line)
