import string

from q3parser import Frag, Suicide, Rename, Teamkill, Messagemode1
from q3parser import LogCreated, OspPlayerStat, OspRoundStart, OspRoundEnd
from q3parser import OspMatchUnpaused
from misc import debug, stripColor, Indexed, warning

class AnalyzeError(Exception): pass

class TopmostError(AnalyzeError):
    """raised when sorting players based on attributes, and no qualified
    players are found"""    
    pass 

## ANALYZER

class Team:        
    def __init__(self, teamname):
        self.teamname = teamname
        self.tag = ''
        self.no_players = 0
        self.frags = 0
        self.ospfrags = 0
        self.suicides = 0
        self.deaths = 0
        self.teamkills = 0
        self.mm1 = 0
        self.effiency = 0

        self.gibs = 0 # etpro only
        
        self.dg = 0
        self.td = 0
        self.dkratio = 0
        self.dr = 0
        self.dg_dr_ratio = 0
        
def makeOSPAsciiName(name):
    asciiName = stripColor(name)[:15].strip()
    asciiName = [x for x in asciiName if x in string.printable]
    return ''.join(asciiName)

class Player:
    def __init__(self, name):
        self.name = name        
        self.asciiName = makeOSPAsciiName(name)
        
        self.team = ""
        self.frags = 0
        self.ospfrags = 0
        
        self.suicides = 0
        self.deaths = 0
        self.teamkills = 0
        self.teamkilled = 0
        self.mm1 = 0

        self.gibs = 0 # etpro only

        self.dg = 0
        self.dr = 0
        self.td = 0
        self.dkratio = 0

        self.smg_frags = 0
        
        self.effiency = 0        
        self.frag_streak = 0        
        self.frag_cur_streak = 0

        self.side = {}
        self.side['blue'] = 0
        self.side['red'] = 0
        
        self.death_streak = 0        
        self.death_cur_streak = 0

        self.players_killed = {}
        self.players_diedby = {}


        self.players_teamkilled = {}
        self.players_teamdiedby = {}
        
        self.weapons_killed = {}
        self.weapons_diedby = {}

class Players:
    def __init__(self):
        self.players = {}
        self.hasOSPInfo = 0
        
    def __getitem__(self, plr):
        return self.players[plr]
    
    def add_plr_if_needed(self, player):
        if not self.players.has_key(player):
            self.players[player] = Player(player)
        
    def add_frag(self, player, victim, weapon):
        self.add_plr_if_needed(player)
        self.add_plr_if_needed(victim)

        plr = self.players[player]
        plr.frags += 1

        plr.frag_cur_streak += 1
        if plr.frag_cur_streak  > plr.frag_streak:
            plr.frag_streak = plr.frag_cur_streak      

        self.players[player].death_cur_streak = 0
        
        self.players[player].players_killed.setdefault(self[victim], 0)
        self.players[player].players_killed[self[victim]] += 1

        self.players[player].weapons_killed.setdefault(weapon, 0)
        self.players[player].weapons_killed[weapon] += 1

        if weapon in ['MP-40', 'Thompson', 'Sten']:
            plr.smg_frags += 1
            
    def add_death(self, player, killer, weapon):
        self.add_plr_if_needed(player)
        self.add_plr_if_needed(killer)
        
        self.players[player].deaths += 1
        self.players[player].frag_cur_streak = 0
        plr = self.players[player]
        
        plr.death_cur_streak += 1
        if plr.death_cur_streak  > plr.death_streak:
            plr.death_streak = plr.death_cur_streak      

        self.players[player].players_diedby.setdefault(self[killer], 0)
        self.players[player].players_diedby[self[killer]] += 1

        self.players[player].weapons_diedby.setdefault(weapon, 0)
        self.players[player].weapons_diedby[weapon] += 1
        
    def add_suicide(self, player):
        self.add_plr_if_needed(player)
        self.players[player].suicides += 1
        self.players[player].frag_cur_streak = 0
        plr = self.players[player]
        
        plr.death_cur_streak += 1
        if plr.death_cur_streak  > plr.death_streak:
            plr.death_streak = plr.death_cur_streak      

    def add_tk(self, player, victim):
        self.add_plr_if_needed(player)
        self.add_plr_if_needed(victim)

        # removed by popular demand
        #self.players[victim].deaths += 1
        self.players[player].teamkills += 1
        self.players[victim].teamkilled += 1
        
        plr = self.players[victim]        
        plr.frag_cur_streak = 0

        plr.death_cur_streak += 1
        if plr.death_cur_streak  > plr.death_streak:
            plr.death_streak = plr.death_cur_streak      
        
        self.players[player].players_teamkilled.setdefault(self[victim], 0)
        self.players[player].players_teamkilled[self[victim]] += 1

        self.players[victim].players_teamdiedby.setdefault(self[player], 0)
        self.players[victim].players_teamdiedby[self[player]] += 1
        
    def add_mm1(self, player):
        self.add_plr_if_needed(player)
        self.players[player].mm1 += 1
        
    def rename(self, before, after):
        self.add_plr_if_needed(before)
        
        if self.players.has_key(after):
            # player played with name 'after', then
            # reconnect, played with some other name, and then
            # renamed itself back to 'after'
            # we have to merge stats
            a = self.players[after]
            b = self.players[before]
            a.frags += b.frags
            a.ospfrags += b.ospfrags
            a.deaths += b.deaths
            a.teamkills += b.teamkills
            a.mm1 += b.mm1
            a.frag_streak = max(a.frag_streak, b.frag_streak)
            a.frag_cur_streak = max(a.frag_cur_streak, b.frag_cur_streak)
            a.death_streak = max(a.death_streak, b.death_streak)
            a.death_cur_streak = max(a.death_cur_streak, b.death_cur_streak)

            dicts = ['players_killed', 'players_diedby',
                     'players_teamkilled', 'players_teamdiedby',
                     'weapons_killed', 'weapons_diedby']

            for dic in dicts:
                for k,v in getattr(b, dic).items():
                    res = getattr(a, dic)
                    res.setdefault(k, 0)
                    res[k] += v
        else:
            self.players[after] = self.players[before]
            self.players[after].name = after
            self.players[after].asciiName = makeOSPAsciiName(after)

        
        del self.players[before]

    def add_osp(self, event):

        for plr in self.players.values():
            if event.player.startswith(plr.asciiName):
                break
        else:
            warning("no player found for osp reported player:%s" %
                    event.player)
            return

        self.hasOSPInfo = 1
        #debug(event.kill, event.round, plr.name, plr.frags)
        plr.ospfrags += event.kill
        plr.dg += event.dg
        plr.dr += event.dr
        plr.td += event.td
        plr.gibs += event.gp

        #print plr.asciiName, plr.frags, o
class Award:
    "Award that player(s) get based on their achievements"
    def __init__(self, title, winners, desc):
        self.title = title
        self.winners = winners
        self.desc = desc

class Q3Analyzer:
    """Analyzes parsed events"""
    def __init__(self, parser_result):
        self.events = parser_result.event_list
        self.is_etpro = parser_result.is_etpro
        
        if len(self.events) <= 2:
            raise AnalyzeError("Too few events detected in this logfile.")
        
        # first anylyze based on events        
        self.analyze_events()

        # 2nd pass resolves teams
        self.resolve_teams()   
        
        # then, inactive player removal, so that player
        # who joined server but didn't do anything wont show up    
        self.remove_inactive_players()
        
        # calculate individual effiency
        self.calculate_effiency()

        # next, tag analysis. 
        self.resolve_tags()
        
        # summarize teamstats
        self.summarize_teamstats()        

        # awards
        self.create_awards()

        # sanity check
        self.sanity_check()

    def sanity_check(self):
        probs = []
        for plr in self.players.players.values():
            if not plr.ospfrags:
                continue
            if plr.frags != plr.ospfrags:
                debug("%s frags=%d, ospfrags=%d" % (plr.name, plr.frags,
                                                      plr.ospfrags))
                if abs(plr.frags / float(plr.ospfrags)-1) > 0.1:
                    probs.append(plr.name)

        if probs:
            warning("Player(s) %s frags differ more than 10%% from "
                    "osp stats. This might be caused by players reconnecting "
                    "at middle of round, or packetloss. This will show as "
                    "somewhat corrupted damage info." %
                    ', '.join(probs))
            
    def create_awards(self):
        
        plr_list = self.players.players.values()
        awards = []

        # good
        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'effiency')
            title = "Terminator"
            desc = "for killing effiency of %s" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass        

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'frags')
            title = "Slaughterhouse"
            desc = "with total of %s kills" % value
            awards.append(Award(title, plrs, desc))        
        except TopmostError:
            pass
        
        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'frag_streak')
            title = "Rampage"
            desc = "for %s frags without dying" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'smg_frags')
            title = "Zen of SMG"
            desc = "for %s frags with Submachine Guns" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        # neutral
        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'deaths',
                                                       reverse=1)
            title = "Sly Fox"
            desc = "for getting killed only %s times" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass        

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'mm1')
            title = "Blabbermouth"
            desc = "for %s lines of mm1" % value
            if value < 10:
                raise TopmostError # too little flood
            
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'dkratio')
            title = "Mr. Pain"
            desc = "for giving damage %s for each frag" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'dkratio',
                                                       reverse=1)
            title = "Fragstealer"
            desc = "for gettings frags for only %s damage per frag" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        # bad
        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'suicides')
            title = "The Dr. Jack"
            desc = "for committing suicide %s times" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass                

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'deaths')
            title = "Slaughterhouse Llama"
            desc = "for getting slaughtered a total of %s times" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'teamkills')
            title = "Friends? We don't need no stinking friends!"
            desc = "for killing %s teammates" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list, 'td')
            title = "Triggerhappy"
            desc = "for giving %s damage to his teammates " \
                   "(all players avg=%s)" % (
                value, self.teams['total'].avg_td)
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        try:
            plrs, value = self.topmost_players_of_attr(plr_list,'death_streak')
            title = "Freefrag"
            desc = "for %s deaths without fragging" % value
            awards.append(Award(title, plrs, desc))
        except TopmostError:
            pass

        if self.is_etpro:
            try:
                plrs, value = self.topmost_players_of_attr(plr_list,'gibs')
                title = "Gravedigger"
                desc = "for %s gibs" % value
                awards.append(Award(title, plrs, desc))
            except TopmostError:
                pass
        
        self.awards = awards
    
    def topmost_players_of_attr(self, plr_list, attribute, reverse=0):
        "Given attribute, return player(s) who have hightest value"  
        plr_list = return_list_sorted_by(plr_list, attribute)

        if reverse:
            plr_list.reverse()

        # return all the players with same topmost value
        attr_list = [getattr(x, attribute) for x in plr_list]

        # empty list
        if len(attr_list) == 0:
            raise TopmostError("empty list") 

        # if highest value is zero, then we ignore it.
        if attr_list[0] == 0:
            raise TopmostError("hi-value is zero")

        # return (list of players), value
        return plr_list[0:attr_list.count(attr_list[0])], attr_list[0]
        
        
    def summarize_teamstats(self):        
        teams = {}
        teams['red'] = Team('red')
        teams['blue'] = Team('blue')
        teams['total'] = Team('Total')
        
        for team in teams.values():

            this_team_players = [x for x in self.players.players.values() 
                                 if x.team == team.teamname]
            
            team.no_players = len(this_team_players)
            
            for plr in this_team_players:
                team.frags += plr.frags
                team.deaths += plr.deaths
                team.teamkills += plr.teamkills
                team.suicides += plr.suicides                
                
                #print plr.frags, "\t", plr.ospfrags, "\t", plr.name
                # XXX: missing rest of osp stuff

                if plr.dr == 0:
                    plr.dg_dr_ratio = "max"
                else:
                    plr.dg_dr_ratio = round(float(plr.dg) / plr.dr,2)
                
                team.ospfrags += plr.ospfrags
                team.dg += plr.dg
                team.td += plr.td
                team.dr += plr.dr
                team.gibs += plr.gibs

            if team.frags == 0:
                team.effiency = 0
            elif team.deaths + team.suicides == 0:
                team.effiency = "max"            
            else:
                team.effiency = round(
                    float(team.frags)/(team.deaths + team.suicides), 2)

            if team.ospfrags == 0:
                team.dkratio = "max"
            else:
                team.dkratio = int(round(team.dg / team.ospfrags))

            if team.dr == 0:
                team.dg_dr_ratio = "max"
            else:
                team.dg_dr_ratio = round(float(team.dg) / team.dr, 2)

        teams['red'].tag = self.red_tag
        teams['blue'].tag = self.blue_tag

        for attr in ['frags', 'deaths', 'teamkills', 'suicides']:
            setattr(teams['total'], attr,
                    getattr(teams['blue'], attr) + 
                    getattr(teams['red'], attr))
                    
        teams['total'].effiency = round(
            float(teams['total'].frags)/(teams['total'].deaths +
                                         teams['total'].suicides), 2)

        teams['total'].avg_td = int(round(
            (teams['red'].td + teams['blue'].td) /
            (teams['red'].no_players + float(teams['blue'].no_players))))

        self.teams = teams

        # weapons
        weapons = {}
        for plr in self.players.players.values():
            for weap, kills in plr.weapons_killed.items():
                weapons.setdefault(weap, 0)
                weapons[weap] += kills

        self.weapons = weapons
        
        # best weaponkills
        w_usage = {}
        best_users = {}
        for weap in weapons:
            w_usage[weap] = []
            for plr in self.players.players.values():
                w_usage[weap].append((plr.weapons_killed.get(weap, 0), plr))
            w_usage[weap].sort()
            best_users[weap] = [x[1] for x in w_usage[weap] if x[0] >=
                                w_usage[weap][-1][0]], w_usage[weap][-1][0]
        
        self.best_weapon_users = best_users
            
    def calculate_effiency(self):
        players = self.players.players
        
        for plr in players.values():            
            if plr.frags == 0:
                plr.effiency = 0
            elif plr.deaths + plr.suicides == 0:
                plr.effiency = "max"
            else:
                plr.effiency = round(
                    float(plr.frags)/(plr.deaths + plr.suicides), 2)

            if plr.dg:
                if plr.ospfrags == 0:
                    plr.dkratio = "max"
                else:
                    plr.dkratio = int(round(plr.dg / float(plr.ospfrags)))

    def remove_reduntant_starts(self, events):
        remove_next_fight = 0
        for event in events[:]:
            if isinstance(event, OspMatchUnpaused):
                remove_next_fight = 1
            if isinstance(event, OspRoundStart) and remove_next_fight:
                events.remove(event)
                remove_next_fight = 0
                debug("removing reduntant fight message")
        return events
    
    def remove_prematch_kills(self, events):
        total = []
        active_segment = []
        maybe_active = 0
        re = 0

        for event in events:
            #print event
            #print re, maybe_active, event
            #if isinstance(event, Messagemode1):
            #    print event.msg
            if isinstance(event, OspRoundStart):
                if maybe_active:
                    active_segment = self.return_nonfrag_events(active_segment)
                maybe_active = 1
                continue
            elif isinstance(event, OspRoundEnd):
                re += 1
                if not maybe_active:
                    warning("Round ended before it started. This logfile "
                            "does not start at begin, or is missing events. "
                            "Events from this partial round are not counted.")
                    continue
                total.extend(active_segment)
                active_segment = []
                maybe_active = 0
                continue

            if maybe_active:
                active_segment.append(event)
            else:
                if self.is_nonfrag_event(event):
                    active_segment.append(event)

        #print "ee"
        #print active_segment
        #return active_segment
        #print "whii"
        #print total
        return total
    
    def return_nonfrag_events(self, events):
        out = []
        dis = 0
        for event in events:
            if self.is_nonfrag_event(event):
                out.append(event)
                continue
            assert not isinstance(event, OspRoundEnd), events
            assert not isinstance(event, OspPlayerStat), events
            dis += 1
        
        debug("discarded %d events" % dis)
        return out

    def is_nonfrag_event(self, event):
        if (isinstance(event, Messagemode1) or
            isinstance(event, Rename)):
            return 1
        else:
            return 0

    def analyze_events(self):
        events = self.events
        
        # first even is always timestamp
        assert isinstance(events[0], LogCreated)
        
        self.timestamp = events[0].timestamp        
        del events[0]                
        
        players = Players()
        mm1 = [] # mm1-events

        # look if events has osp events
        #print "before", len(events)
        if len([x for x in events if isinstance(x, OspRoundStart)]):
            # it has, so lets do prematch kills-removal
            events = self.remove_reduntant_starts(events)
            events = self.remove_prematch_kills(events)

        
        #print "after", len(events)
        re = rs = 0
        for event in events:
            # FRAG
            if isinstance(event, Frag):
                players.add_frag(event.attacker, event.victim, event.weapon)
                players.add_death(event.victim, event.attacker, event.weapon)

            # SUICIDE
            elif isinstance(event, Suicide):
                players.add_suicide(event.depressed)

            # TEAMKILL
            elif isinstance(event, Teamkill):
                players.add_tk(event.attacker, event.victim)

            # RENAME
            elif isinstance(event, Rename):
                players.rename(event.before, event.after)

            # MM1
            elif isinstance(event, Messagemode1):
                players.add_mm1(event.talker)
                mm1.append(event)

            # OSP
            elif isinstance(event, OspPlayerStat):
                #print "osp event"
                # if player has participated in next round dont count
                if not self.is_reduntant(event, events):
                    #debug("counting this, round=", event.round)
                    players.add_osp(event)
                else:
                    #debug("not counting, round=", event.round)
                    pass
                
            elif isinstance(event, OspRoundEnd):
                re += 1
                
            elif isinstance(event, OspRoundStart):
                rs += 1
            elif isinstance(event, OspMatchUnpaused):
                pass
            else:
                raise AnalyzeError, "Unknown event: %s" % event

        self.mm1 = mm1
        self.players = players

    def is_reduntant(self, event, events):
        # test only meaningful if round is 1
        if event.round != 1:
            return 0
        # scan forward until osp events found
        i = events.index(event)
        l = len(events)
        # first, go to end of this round stats
        while i < l:
            if not isinstance(events[i], OspPlayerStat):
                    break
            i += 1

        # then beginning of next round
        while i < l:
            if isinstance(events[i], OspPlayerStat):
                    break
            i += 1

        if i >= l:
            return 0

        while i < l:
            if not isinstance(events[i], OspPlayerStat):
                return 0
            if events[i].player == event.player and events[i].round == 2:
                return 1            
            i += 1
            
        return 0
        
    def players_not_in_team(self):
        return [x for x in self.players.players.values() if x.team == '']

    def resolve_teams(self):
        players = self.players.players
        unteamed = self.players_not_in_team()

        # set team of player who has fragged the most
        try:
            topFragger = self.topmost_players_of_attr(unteamed, 'frags')[0][0]
        except TopmostError:
            raise AnalyzeError("There are zero frags in this logfile. Frags "
                               "must happen between round start and end "
                               "messages in order to be counted.")
            
        topFragger.team = 'blue' # red would do just fine too

        k = 3
        while 1:            
            before = len(self.players_not_in_team())
            self.resolve_team_iteration()
            
            for player in self.players_not_in_team():
                if player.side['red'] > (player.side['blue']*k):
                    player.team = 'red'
                elif player.side['blue'] > (player.side['red']*k):
                    player.team = 'blue'
                    
            plrs_not_in_team = self.players_not_in_team()
            after = len(self.players_not_in_team())

            # if no new player teams were resolved, quit loop
            if after == 0 or before == after:
                break
            
            assert before > after
            if k > 1:
                k -= 1

        for plr in self.players_not_in_team():
            warning("could not find team for: %s" % plr.name)
            del players[plr.name]                
        
    def resolve_team_iteration(self):
        players = self.players.players        
        teamed_players = [x for x in players.values() if x.team != '']

        for player in teamed_players:
            assert player.team in ['blue', 'red']

            if player.team == 'blue':
                opposite = 'red'
            else:
                opposite = 'blue'
                
            for plr in player.players_killed.keys():
                plr.side[opposite] += player.players_killed[plr]

            for plr in player.players_teamkilled.keys():
                plr.side[player.team] += player.players_teamkilled[plr]

            for plr in player.players_diedby.keys():
                plr.side[opposite] += player.players_diedby[plr]

            for plr in player.players_teamdiedby.keys():
                plr.side[player.team] += player.players_teamdiedby[plr]

    def resolve_tags(self):
        players = self.players.players
        
        blue_names = [stripColor(x.name)
                      for x in players.values() if x.team == 'blue']
        red_names = [stripColor(x.name)
                     for x in players.values() if x.team == 'red']

        blue_prefix = self.find_most_common_prefix(blue_names)
        red_prefix = self.find_most_common_prefix(red_names)

        # we can find suffix by reversing player names and calling
        # find_prefix, and then reversing result.
        rev_blue = []
        rev_red = []
        
        for name in blue_names:
            rev = list(name)
            rev.reverse()
            rev_blue.append(''.join(rev))

        for name in red_names:
            rev = list(name)
            rev.reverse()
            rev_red.append(''.join(rev))

        blue_suffix = self.find_most_common_prefix(rev_blue)        
        red_suffix = self.find_most_common_prefix(rev_red)

        rev = list(blue_suffix)
        rev.reverse()
        blue_suffix = ''.join(rev)

        rev = list(red_suffix)
        rev.reverse()
        red_suffix = ''.join(rev)
        
        # choose between prefix and suffix, longest wins        
        if len(blue_prefix) >= len(blue_suffix):
            blue_tag = blue_prefix
        else:
            blue_tag = blue_suffix

                    
        if len(red_prefix) >= len(red_suffix):
            red_tag = red_prefix
        else:
            red_tag = red_suffix


        # ok one quich haxor, sometimes after weirdcharacters ^ is appended,
        # so we remove it
        if len(red_tag) >= 1:
            if red_tag[-1] == '^': red_tag = red_tag[:-1]

        if len(blue_tag) >= 1:
            if blue_tag[-1] == '^': blue_tag = blue_tag[:-1]

        # remove common characters in tag
        # eg. [sca.wizz -> [sca.  so starting '[' and ending '.'
        # are not part of the tag.
        # however, dont perform removing if tag shortens too much
        #
        removeChrs = ('.', '[', ']', '{', '}', '(', ')', '*', '-')            
        while len(red_tag) >= 3 and red_tag[0] in removeChrs:
            red_tag = red_tag[1:]
        while len(red_tag) >= 3 and red_tag[-1] in removeChrs:
            red_tag = red_tag[:-1]

        while len(blue_tag) >= 3 and blue_tag[0] in removeChrs:
            blue_tag = blue_tag[1:]
        while len(blue_tag) >= 3 and blue_tag[-1] in removeChrs:
            blue_tag = blue_tag[:-1]
            
        self.red_tag, self.blue_tag = red_tag, blue_tag


        
    def find_most_common_prefix(self, names):
        # find the most likely prefix from string.
        # i wrote this sometime ago, and now i dont have any clue
        # how it does it stuff. seems to work, thou.
        #
        if len(names) == 0:
            return ''
        
        chars = [{}]   # dictionary of (char, count) pairs
        max_name_len = max([len(x) for x in names])

        prefix = []
  
        for i in range(0, max_name_len):            
            for name in names:                
                if i < len(name):
                    cur_char = name[i]
                    if not chars[i].has_key(cur_char):
                        chars[i][cur_char] = 1
                    else:
                        chars[i][cur_char] += 1
            chars.append({})

        # sort out (char, count) dictionary, so that char which has
        # most occurances is first. This char is appended to prefix
        # list.
        
        for dic in chars:
            letters = dic.items()            
            letters.sort(lambda x,y : cmp(y[1], x[1]))           
            if len(letters):
                prefix.append(letters[0])
                
        
        # append all prefix characters that 6 out of 9 players
        # had. This should prevend from situations like common char is
        # included in prefix. Example:        
        # >M< Alfa
        # >M< Aatu
        # >M< Arttu
        #
        # with this kind of input data, prefix would be >M< A
        # But I hope that 6/9 is enough strict requirement.         
        
        common =''
        for pre in prefix:
            if float(pre[1])/len(names) >= float(6)/9:               
                common += pre[0]
            else:
                break       

        # last, remove spaces from it
        return common.strip()

    def remove_inactive_players(self):        
        "remove players who didn't do anything"
        players = self.players.players
        for player in players.values():
            if self.is_inactive(player):                
                debug("removing inactive player %s" % player.name)
                del players[player.name]
    
    def is_inactive(self, player):
        if player.frags == 0 and player.teamkills == 0 and \
           player.deaths == 0 and player.suicides == 0:
            return 1
        else:
            return 0            
            
        
def return_list_sorted_by(obj_list, attr):
    """Given list of objects, sorts them according to attribute. Greatest
    value is sorted to first"""    
    obj_list.sort(lambda x, y, attr=attr: cmp(getattr(y, attr),
                                              getattr(x, attr)))
    return obj_list
