<?php
/**
 *    This file is part of "Liberty Gaming NQ Stats Analyzer".
 *
 *    "Liberty Gaming NQ Stats Analyzer" is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    "Liberty Gaming NQ Stats Analyzer" is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * Class PCPIN_GameClientKill
 * Manage kills
 * @author Konstantin Reznichak <k.reznichak@pcpin.com>
 * @copyright Copyright &copy; 2007, Konstantin Reznichak
 */
class PCPIN_GameClientKill extends PCPIN_Session {

  /**
   * Game ID
   * @var int
   */
  var $game_id=0;

  /**
   * Killer GUID
   * @var string
   */
  var $guid='';

  /**
   * Killer class
   * @var int
   */
  var $class=0;

  /**
   * Victim GUID
   * @var string
   */
  var $victim_guid='';

  /**
   * Team
   * @var int
   */
  var $team=0;

  /**
   * Flag: Killed enemy ("y") or teammate ("n")
   * @var string
   */
  var $enemy='';

  /**
   * MOD (mean of death)
   * @var string
   */
  var $mod='';

  /**
   * Kills count
   * @var int
   */
  var $count=0;



  /**
   * Constructor
   * @param   object  &$sessionhandler  Session handler
   */
  function PCPIN_GameClientKill(&$sessionhandler) {
    // Init object
    $this->_s_init($sessionhandler, $this);
  }


  /**
   * Store new record
   * @param   int       $game_id        Game ID
   * @param   string    $guid           Killer GUID
   * @param   int       $class          Killer class
   * @param   string    $victim_guid    Victim GUID
   * @param   int       $team           Team
   * @param   string    $enemy          "y" if victim was an enemy, "n" otherwise
   * @param   string    $mod            MOD
   * @param   int       $count          Kills count
   */
  function addRecord($game_id, $guid, $class, $victim_guid, $team, $enemy, $mod, $count) {
    $this->game_id=$game_id;
    $this->guid=$guid;
    $this->class=$class;
    $this->victim_guid=$victim_guid;
    $this->team=$team;
    $this->enemy=$enemy;
    $this->mod=$mod;
    $this->count=$count;
    $this->_db_insertObj();
  }


  /**
   * Get kill stats grouped by MOD
   * @param   string    $guid         GUID
   * @param   array     $skip_mods    Optional. MODs that shall not be included
   * @param   int       $game_id      Optional. Game ID
   * @return array
   */
  function getKillsMOD($guid, $skip_mods=array(), $game_id=0) {
    $mods=array();
    $query=$this->_db_makeQuery(1070, $guid, $skip_mods, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $mods[]=$data;
    }
    $this->_db_freeResult($result);
    return $mods;
  }


  /**
   * Get kill stats grouped by GUID, Team and MOD
   * @param   int       $game_id      Game ID
   * @return array
   */
  function getKillsGuidTeamMod($game_id) {
    $list=array();
    $query=$this->_db_makeQuery(1075, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $list[]=$data;
    }
    $this->_db_freeResult($result);
    return $list;
  }


  /**
   * Count how many times player was killed by other player
   * @param   string    $killer_guid        Killer GUID
   * @param   string    $victim_guid        Victim GUID
   * @return int
   */
  function countKillsByGuid($killer_guid, $victim_guid) {
    $kills=0;
    $query=$this->_db_makeQuery(1110, $killer_guid, $victim_guid);
    $result=$this->_db_query($query);
    if ($data=$this->_db_fetch($result)) {
      $kills=$data['Kills'];
    }
    $this->_db_freeResult($result);
    return $kills;
  }


  /**
   * Get top N "favorite victims"
   * @param   string    $guid         GUID
   * @param   int       $limit        Limit
   * @return array
   */
  function getFavoriteVictims($guid, $limit) {
    $list=array();
    $query=$this->_db_makeQuery(1090, $guid, $limit);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      // Get deaths
      $data['Deaths']=$this->countKillsByGuid($data['VictimGUID'], $guid);
      $data['Efficiency']=($data['Kills']>0 || $data['Deaths']>0)? 100*$data['Kills']/($data['Kills']+$data['Deaths']) : 0;
      $list[]=$data;
    }
    $this->_db_freeResult($result);
    return $list;
  }


  /**
   * Get top N worst enemies
   * @param   string    $guid         GUID
   * @param   int       $limit        Limit
   * @return array
   */
  function getWorstEnemies($guid, $limit) {
    $list=array();
    $query=$this->_db_makeQuery(1100, $guid, $limit);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      // Get kills
      $data['Kills']=$this->countKillsByGuid($guid, $data['KillerGUID']);
      $data['Efficiency']=($data['Kills']>0 || $data['Deaths']>0)? 100*$data['Kills']/($data['Kills']+$data['Deaths']) : 0;
      $list[]=$data;
    }
    $this->_db_freeResult($result);
    return $list;
  }


  /**
   * Get kills count grouped by team/class
   * @param   string    $killer_guid  GUID
   * @param   int       $team         Optional: Team
   * @param   int       $class        Optional: Class
   * @return array
   */
  function countKillsByTeamClass($killer_guid, $team=null, $class=null) {
    $kills=array();
    $query=$this->_db_makeQuery(1120, $killer_guid, $team, $class);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      if (!isset($kills[$data['Team']])) {
        $kills[$data['Team']]=array();
      }
      if (!isset($kills[$data['Team']][$data['Class']])) {
        $kills[$data['Team']][$data['Class']]=array();
      }
      if (!isset($kills[$data['Team']][$data['Class']][$data['MOD']])) {
        $kills[$data['Team']][$data['Class']][$data['MOD']]=0;
      }
      $kills[$data['Team']][$data['Class']][$data['MOD']]+=$data['Count'];
    }
    $this->_db_freeResult($result);
    return $kills;
  }


  /**
   * Get kills for all players for specified game
   * @param   int       $game_id      Game ID
   * @return array
   */
  function getKillsByTeam($game_id) {
    $kills=array();
    $query=$this->_db_makeQuery(1125, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $kills[]=$data;
    }
    $this->_db_freeResult($result);
    return $kills;
  }


  /**
   * Get GUIDKey+Name and kills number of the player who made most kills with specified MOD
   * @param   string    $mods           Single MOD or multiple mods separated by comma
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestModKiller($mods, $min_game_id=0) {
    $out=array();
    $mods_array=explode(',', str_replace(' ', '', $mods));
    foreach ($mods_array as $key=>$val) {
      $val=trim($val);
      if ($val!='') {
        $mods_array[$key]=$val;
      } else {
        unset($mods_array[$key]);
      }
    }
    $query=$this->_db_makeQuery(1200, $mods_array, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


}
?>