<?php
/**
 *    This file is part of "Liberty Gaming NQ Stats Analyzer".
 *
 *    "Liberty Gaming NQ Stats Analyzer" is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    "Liberty Gaming NQ Stats Analyzer" is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * Class PCPIN_GameClient
 * Manage game clients (link between guid and game)
 * @author Konstantin Reznichak <k.reznichak@pcpin.com>
 * @copyright Copyright &copy; 2007, Konstantin Reznichak
 */
class PCPIN_GameClient extends PCPIN_Session {

  /**
   * Game ID
   * @var int
   */
  var $game_id=0;

  /**
   * GUID
   * @var string
   */
  var $guid='';

  /**
   * Time spent playing AXIS team (seconds)
   * @var int
   */
  var $time_axis=0;

  /**
   * Time spent playing ALLIES team (seconds)
   * @var int
   */
  var $time_allies=0;

  /**
   * Time spent as SPECTATOR (seconds)
   * @var int
   */
  var $time_spectator=0;

  /**
   * Total killed enemies count
   * @var int
   */
  var $killed_enemies=0;

  /**
   * Total killed teammates count
   * @var int
   */
  var $killed_teammates=0;

  /**
   * Total deaths count caused by enemies
   * @var int
   */
  var $killed_by_enemies=0;

  /**
   * Total deaths count caused by teammates
   * @var int
   */
  var $killed_by_teammates=0;

  /**
   * Total deaths count caused by world
   * @var int
   */
  var $killed_by_world=0;

  /**
   * Total suicides count
   * @var int
   */
  var $suicides=0;

  /**
   * Score (XP) when the game ended
   * @var int
   */
  var $score=0;

  /**
   * Ping (ms) when the game ended
   * @var int
   */
  var $ping=0;

  /**
   * Total revives count
   * @var int
   */
  var $revives=0;

  /**
   * Total health packs given
   * @var int
   */
  var $health_given=0;

  /**
   * Total ammo packs given
   * @var int
   */
  var $ammo_given=0;

  /**
   * Total health packs taken
   * @var int
   */
  var $health_taken=0;

  /**
   * Total ammo packs taken
   * @var int
   */
  var $ammo_taken=0;

  /**
   * Dynamite plants
   * @var int
   */
  var $dynamite_plant=0;

  /**
   * Dynamite diffuses
   * @var int
   */
  var $dynamite_diffuse=0;

  /**
   * Repairs
   * @var int
   */
  var $repairs=0;

  /**
   * Disguises
   * @var int
   */
  var $disguises=0;

  /**
   * Disguises
   * @var int
   */
  var $objectives_taken=0;

  /**
   * Kill sprees count (5 or more kills)
   * @var int
   */
  var $kill_sprees=0;

  /**
   * Longest kill spree
   * @var int
   */
  var $longest_kill_spree=0;

  /**
   * Death sprees count (10 or more deaths)
   * @var int
   */
  var $death_sprees=0;

  /**
   * Longest death spree
   * @var int
   */
  var $longest_death_spree=0;

  /**
   * Flag: "y" if player did first kill in match, "n" otherwize
   * @var string
   */
  var $first_kill='n';

  /**
   * Flag: "y" if player had first death in match,"n" otherwize
   * @var string
   */
  var $first_death='n';

  /**
   * Total bounties count
   * @var int
   */
  var $bounties=0;

  /**
   * Killed with engineer-only weapons enemies count
   * @var int
   */
  var $engineer_kills=0;

  /**
   * Killed with fieldops-only weapons enemies count
   * @var int
   */
  var $fieldops_kills=0;

  /**
   * Killed with covertops-only weapons enemies count
   * @var int
   */
  var $covertops_kills=0;





  /**
   * Constructor
   * @param   object  &$sessionhandler  Session handler
   */
  function PCPIN_GameClient(&$sessionhandler) {
    // Init object
    $this->_s_init($sessionhandler, $this);
  }


  /**
   * Add new game client
   * @param   int     $game_id              Game ID
   * @param   string  $guid                 GUID
   * @param   int     $time_axis            Time spent playing AXIS team
   * @param   int     $time_allies          Time spent playing ALLIES team
   * @param   int     $time_spectator       Time spent as SPECTATOR
   * @param   int     $killed_enemies       Killed enemies
   * @param   int     $killed_teammates     Killed teammates
   * @param   int     $killed_by_enemies    Deaths caused by enemies
   * @param   int     $killed_by_teammates  Deaths caused by teammates
   * @param   int     $killed_by_world      Deaths caused by world
   * @param   int     $suicides             Suicides
   * @param   int     $score                Score (XP)
   * @param   int     $ping                 Ping (ms)
   * @param   int     $revives              Revives count
   * @param   int     $health_given         Given health packs count
   * @param   int     $ammo_given           Given ammo packs count
   * @param   int     $health_taken         Taken health packs count
   * @param   int     $ammo_taken           Taken ammo packs count
   * @param   int     $dynamite_plant       Dynamite plants
   * @param   int     $dynamite_diffuse     Dynamite diffuses
   * @param   int     $repairs              Repairs
   * @param   int     $disguises            Disguises
   * @param   int     $objectives_taken     Taken objectives
   * @param   string  $first_kill           "first_kill" award
   * @param   string  $first_death          "first_death" award
   * @param   int     $kill_sprees          Kill sprees count
   * @param   int     $longest_kill_spree   Longest kill spree
   * @param   int     $death_sprees         Death sprees count
   * @param   int     $longest_death_spree  Longest death spree
   * @param   int     $bounties             Bounties count
   * @param   int     $engineer_kills       Killed with engineer-only weapons enemies count
   * @param   int     $fieldops_kills       Killed with fieldops-only weapons enemies count
   * @param   int     $covertops_kills      Killed with covertops-only weapons enemies count
   */
  function addClient($game_id,
                     $guid,
                     $time_axis,
                     $time_allies,
                     $time_spectator,
                     $killed_enemies,
                     $killed_teammates,
                     $killed_by_enemies,
                     $killed_by_teammates,
                     $killed_by_world,
                     $suicides,
                     $score,
                     $ping,
                     $revives,
                     $health_given,
                     $ammo_given,
                     $health_taken,
                     $ammo_taken,
                     $dynamite_plant,
                     $dynamite_diffuse,
                     $repairs,
                     $disguises,
                     $objectives_taken,
                     $first_kill,
                     $first_death,
                     $kill_sprees,
                     $longest_kill_spree,
                     $death_sprees,
                     $longest_death_spree,
                     $bounties,
                     $engineer_kills,
                     $fieldops_kills,
                     $covertops_kills
                     ) {
    $this->game_id=$game_id;
    $this->guid=$guid;
    $this->time_axis=$time_axis;
    $this->time_allies=$time_allies;
    $this->time_spectator=$time_spectator;
    $this->killed_enemies=$killed_enemies;
    $this->killed_teammates=$killed_teammates;
    $this->killed_by_enemies=$killed_by_enemies;
    $this->killed_by_teammates=$killed_by_teammates;
    $this->killed_by_world=$killed_by_world;
    $this->suicides=$suicides;
    $this->score=$score;
    $this->ping=$ping;
    $this->revives=$revives;
    $this->health_given=$health_given;
    $this->ammo_given=$ammo_given;
    $this->health_taken=$health_taken;
    $this->ammo_taken=$ammo_taken;
    $this->dynamite_plant=$dynamite_plant;
    $this->dynamite_diffuse=$dynamite_diffuse;
    $this->repairs=$repairs;
    $this->disguises=$disguises;
    $this->objectives_taken=$objectives_taken;
    $this->first_kill=$first_kill;
    $this->first_death=$first_death;
    $this->kill_sprees=$kill_sprees;
    $this->longest_kill_spree=$longest_kill_spree;
    $this->death_sprees=$death_sprees;
    $this->longest_death_spree=$longest_death_spree;
    $this->bounties=$bounties;
    $this->engineer_kills=$engineer_kills;
    $this->fieldops_kills=$fieldops_kills;
    $this->covertops_kills=$covertops_kills;
    $this->_db_insertObj();
  }


  /**
   * Get player list with stats
   * @param   array     $exclude_guids    Optional: Array with GUIDs to be excluded from list. IGNORED, if $player_key is not empty
   * @param   string    $name             Optional: Player name
   * @param   string    $sort_by          Optional: Sort by ...
   * @param   string    $sort_dir         Optional: Sort dir ("asc" or "desc")
   * @param   int       $limitstart       Optional: Start from N'th result
   * @param   int       $limit            Optional: Max results to return
   * @param   string    $player_key       Optional: MD5-Encoded GUID
   * @param   int       $game_id          Optional: Game ID
   * @return array
   */
  function listPlayers($exclude_guids='', $name='', $sort_by='', $sort_dir='', $limitstart=0, $limit=0, $player_key='', $game_id=0) {
    $players=array();
    $only_guids=array();
    $clientname=new PCPIN_ClientName($this);
    $do_list=true;
    $total_results=0;
    if (!empty($name)) {
      // Get corresponding GUIDs
      $clientname=new PCPIN_ClientName($this);
      if ($clientname->_db_getList('guid', 'name_txt _LIKE %'.$name.'%')) {
        foreach ($clientname->_db_list as $data) {
          $only_guids[$data['guid']]=1;
          $clientname->_db_freeList();
        }
      }
      $only_guids=array_keys($only_guids);
      if (empty($only_guids)) {
        $do_list=false;
      }
    }
    if ($do_list) {
      $query=$this->_db_makeQuery(1000, $exclude_guids, $only_guids, $sort_by, $sort_dir, $limitstart, $limit, $player_key, $game_id);
      $result=$this->_db_query($query);
      while ($data=$this->_db_fetch($result)) {
        // Get used names
        $data['UsedNames']=$clientname->getUsedNames($data['GUID']);
        $players[]=$data;
      }
      $this->_db_freeResult($result);
      // Get total results count
      $query=$this->_db_makeQuery(1005, $exclude_guids, $only_guids, '', '', 0, 0, $player_key, $game_id);
      $result=$this->_db_query($query);
      $data=$this->_db_fetch($result);
      $this->_db_freeResult($result);
      $total_results=$data['TotalResults'];
    }
    return array('TotalResults'=>$total_results,
                 'ListedResults'=>$players);
  }


  /**
   * Get average ping
   * @param   string    $guid       GUID
   * @param   int       $game_id    Optional: Game ID
   * @return float
   */
  function getAvgPing($guid, $game_id=0) {
    $ping=0.0;
    $query=$this->_db_makeQuery(1030, $guid, $game_id);
    $result=$this->_db_query($query);
    if ($data=$this->_db_fetch($result)) {
      $ping=$data['AvgPing'];
    }
    $this->_db_freeResult($result);
    return $ping;
  }


  /**
   * Get GUIDKey+Name and kills number of the player who achieved the longest kill spree
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getLongestKillSpree($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1220, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get GUIDKey+Name and kills number of the player who achieved the most kill sprees
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getMostKillSprees($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1225, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get GUIDKey+Name and bounties number of the player who collected most bounty points
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestBountyHunter($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1230, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get player with the best Kills/Deaths ratio
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestKD($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1240, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get player with the best Kills/game ratio
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestKG($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1250, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get GUIDKey+Name and deaths number of the player who achieved the longest deaths spree
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getLongestDeathSpree($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1220, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get "best medic" player
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestMedic($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1270, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get "best engineer" player
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestEngineer($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1280, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get "best field op" player
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestFieldOp($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1290, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }


  /**
   * Get "best covert op" player
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestCovertOp($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1300, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
      $this->_db_freeResult($result);
    }
    return $out;
  }




}
?>