<?php
/**
 *    This file is part of "Liberty Gaming NQ Stats Analyzer".
 *
 *    "Liberty Gaming NQ Stats Analyzer" is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    "Liberty Gaming NQ Stats Analyzer" is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * Class PCPIN_GameClientDeath
 * Manage world deaths (for example, crused)
 * @author Konstantin Reznichak <k.reznichak@pcpin.com>
 * @copyright Copyright &copy; 2007, Konstantin Reznichak
 */
class PCPIN_GameClientDeath extends PCPIN_Session {

  /**
   * Game ID
   * @var int
   */
  var $game_id=0;

  /**
   * GUID
   * @var string
   */
  var $guid='';

  /**
   * Killer GUID
   * @var string
   */
  var $killer_guid='';

  /**
   * Team
   * @var int
   */
  var $team=0;

  /**
   * Flag: Killed by enemy ("y"), teammate ("n") or world ("w")
   * @var string
   */
  var $enemy='';

  /**
   * MOD (mean of death)
   * @var string
   */
  var $mod='';

  /**
   * Deaths count
   * @var int
   */
  var $count=0;



  /**
   * Constructor
   * @param   object  &$sessionhandler  Session handler
   */
  function PCPIN_GameClientDeath(&$sessionhandler) {
    // Init object
    $this->_s_init($sessionhandler, $this);
  }


  /**
   * Store new record
   * @param   int       $game_id        Game ID
   * @param   string    $guid           GUID
   * @param   string    $killer_guid    Killer GUID
   * @param   int       $team           Team
   * @param   string    $enemy          Enemy flag
   * @param   string    $mod            MOD
   * @param   int       $count          Kills count
   */
  function addRecord($game_id, $guid, $killer_guid, $team, $enemy, $mod, $count) {
    $this->game_id=$game_id;
    $this->guid=$guid;
    $this->killer_guid=$killer_guid;
    $this->team=$team;
    $this->enemy=$enemy;
    $this->mod=$mod;
    $this->count=$count;
    $this->_db_insertObj();
  }


  /**
   * Get death stats grouped by MOD
   * @param   string    $guid         GUID
   * @param   array     $skip_mods    Optional. MODs that shall not be included
   * @param   int       $game_id      Optional. Game ID
   * @return array
   */
  function getDeathsMOD($guid, $skip_mods=array(), $game_id=0) {
    $mods=array();
    $query=$this->_db_makeQuery(1080, $guid, $skip_mods, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $mods[]=$data;
    }
    $this->_db_freeResult($result);
    return $mods;
  }


  /**
   * Get death stats grouped by GUID, Team and MOD
   * @param   int       $game_id      Game ID
   * @return array
   */
  function getDeathsGuidTeamMod($game_id) {
    $list=array();
    $query=$this->_db_makeQuery(1085, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $list[]=$data;
    }
    $this->_db_freeResult($result);
    return $list;
  }


  /**
   * Get world death for specified game count grouped by GUID team and MOD
   * @param   int     $game_id    Game ID
   * @return array
   */
  function getWorldDeathsTeamMOD($game_id) {
    $deaths=array();
    $query=$this->_db_makeQuery(1140, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $deaths[]=$data;
    }
    $this->_db_freeResult($result);
    return $deaths;
  }


  /**
   * Get deaths for all players for specified game
   * @param   int       $game_id      Game ID
   * @return array
   */
  function getDeathsByTeam($game_id) {
    $deaths=array();
    $query=$this->_db_makeQuery(1127, $game_id);
    $result=$this->_db_query($query);
    while ($data=$this->_db_fetch($result)) {
      $deaths[]=$data;
    }
    $this->_db_freeResult($result);
    return $deaths;
  }


  /**
   * Get deaths count for all MODs
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function countDeathsByMod($min_game_id=0) {
    $mods=array();
    $query=$this->_db_makeQuery(1190, $min_game_id);
    if ($result=$this->_db_query($query)) {
      while ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $mods[]=$data;
      }
      $this->_db_freeResult($result);
    }
    return $mods;
  }


  /**
   * Get GUIDKey+Name and deaths number of the player who got most deaths with specified MOD
   * @param   string    $mod            MOD
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getBestModVictim($mod, $min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1210, $mod, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
    }
    return $out;
  }


  /**
   * Get player with most world deaths number
   * @param   int       $min_game_id    Optional. Minimum game ID
   * @return  array
   */
  function getMostWorldDeaths($min_game_id=0) {
    $out=array();
    $query=$this->_db_makeQuery(1260, $min_game_id);
    if ($result=$this->_db_query($query)) {
      if ($data=$this->_db_fetch($result, MYSQL_ASSOC)) {
        $cn=new PCPIN_ClientName($this);
        if ($cn->_db_getList('name, name_txt', 'guid = '.$data['GUID'], 'last_used = y', 1)) {
          $data['Name']=$cn->_db_list[0]['name'];
          $data['NameTxt']=$cn->_db_list[0]['name_txt'];
          $cn->_db_freeList();
          $out=$data;
        }
      }
    }
    return $out;
  }



}
?>