package etstatistics;

/**
ET-Statistics
Copyright © 2009-10  Anthales

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

import java.io.BufferedWriter;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.FileReader;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.text.DecimalFormat;
import static java.lang.Integer.parseInt;

public class ETStatistics {
	/** List of all unique weapon names */
	private static final String[] WEAPONS = {
		"Knife",
		"Luger",	"Colt",
		"MP-40",	"Thompson",

		"K43 Rifle", "Garand",
		"Sten",
		"FG-42",

		"Grenade",
		"G.Launchr",
		"Landmine",
		"Satchel",

		"Panzer",
		"Mortar",
		"MG-42 Gun",
		"F.Thrower",

		"Airstrike",
		"Artillery",

		"Dynamite",
		"Syringe",
		"SmokeScrn",
	};

	public static final int WEAPON_COUNT = WEAPONS.length;

	public static String getWeapon(int i) {
		return WEAPONS[i];
	}


	/** List of all stats */
	private ArrayList<Statistic> stat = new ArrayList<Statistic>();
	/** Unified stat */
	private Statistic finalStat = null;

	private static final String PMAP      = ">>> Map: ";
	private static final String PNICK     = "Overall stats for: ";
	private static final String PDMGGIVE  = "Damage Given: ";
	private static final String PTDMGGIVE = "Team Damage: ";
	private static final String PDMGRECV  = "Damage Recvd: ";
	private static final String PTDMGRECV = "Team Damage Received: ";
	private static final String PGIB      = "        Gibs: ";
	private static final String PRANK     = "Rank: ";

	/** 
	 * Parses a file, adding it to the stats list, when it was read
	 * successfully.
	 */
	public boolean add(File file) throws IOException {
		BufferedReader bf = new BufferedReader(new FileReader(file));

		Statistic s = new Statistic();

		try {
			//Read each line
			while (true) {
				String data = bf.readLine();
				if (data == null) return false;   //EOF before expected
				if (data.length() == 0) continue; //Skip empty lines

				//Parse mapname
				if (data.startsWith(PMAP)) {
					s.map = data.substring(PMAP.length());

				//Parse player nick and amount of rounds played
				} else if (data.startsWith(PNICK)) {
					int sub  = data.lastIndexOf("(");
					int sub2 = data.lastIndexOf(" Round");
					s.playerName = data.substring(PNICK.length(), sub);
					s.rounds = parseInt(data.substring(sub+1, sub2));

				//Parse weapon statistics
				} else if (data.indexOf(":") == 9) {
					WeaponStatistic tempWS = new WeaponStatistic();

					tempWS.weaponName = data.substring(0, 9).trim();

					//Accuracy (will be calculated)
//					temp = data.substring(11, 16).trim();
//					if (temp.length() > 0) tempWS.acc = Float.parseFloat(temp);
//					else                   tempWS.acc = -1;

					//Amount of hits
					String temp = data.substring(17, 21).trim();
					if (temp.length() > 0) tempWS.hits = parseInt(temp);
					else                   tempWS.hits = 0;

					//Amount of shots
					temp = data.substring(22, 26).trim();
					if (temp.length() > 0) tempWS.shots = parseInt(temp);
					else                   tempWS.shots = 0;

					//Amount of frags
					temp = data.substring(27, 32).trim();
					if (temp.length() > 0) tempWS.kills = parseInt(temp);
					else                   tempWS.kills = 0;

					//Amount of player deaths by this weapon
					temp = data.substring(33, 39).trim();
					if (temp.length() > 0) tempWS.deaths = parseInt(temp);
					else                   tempWS.deaths = 0;

					//Amount of head shots
					if (data.length() >= 40) {
						temp = data.substring(40, 49).trim();
						tempWS.headshots = parseInt(temp);
					} else
						tempWS.headshots = 0;

					//Add weapon statistic to the actual statistic
					s.weaponStats.add(tempWS);
					s.kills  += tempWS.kills;
					s.deaths += tempWS.deaths;
//					s.eff    += (double)tempWS.kills / tempWS.deaths;
					s.eff     = (double)s.kills / s.deaths;

				//Parse damage dealt/received and amount of gibs
				} else if (data.startsWith(PDMGGIVE)) {
					int sub = data.indexOf(PTDMGGIVE);
					s.damageGiven = parseInt(
						data.substring(PDMGGIVE.length(), sub).trim());
					s.teamDamageGiven = parseInt(
						data.substring(sub + PTDMGGIVE.length()).trim());
				} else if (data.startsWith(PDMGRECV)) {
					int sub = data.indexOf(PTDMGRECV);
					s.damageReceived = parseInt(
						data.substring(PDMGRECV.length(), sub).trim());
					s.teamDamageReceived = parseInt(
						data.substring(sub + PTDMGRECV.length()).trim());
				} else if (data.startsWith(PGIB)) {
					s.gibs = parseInt(
						data.substring(PGIB.length()).trim());
				}

				//Parse rank and experience points
				else if (data.startsWith(PRANK)) {
					int sub = data.lastIndexOf("(");
					s.rank = data.substring(PRANK.length(), sub).trim();
					int sub2 = data.lastIndexOf("XP");
					s.experience = parseInt(data.substring(sub+1, sub2).trim());
				}

				//End parsing
				else if (data.startsWith("Skills         "))
					break;
			}
		} catch (NumberFormatException e) {
			return false;
		} catch (IndexOutOfBoundsException e) {
			return false;
		}

		stat.add(s);
		return true;
	}

	/** Returns the list of all statistics */
	public ArrayList<Statistic> getStats() {
		return stat;
	}

	/** Returns the unified statistic, if calculated. */
	public Statistic getFinalStat() {
		return finalStat;
	}

	private static DecimalFormat df = new DecimalFormat("0.00");
	/** Rounds to two decimal places. */
	public static String roundCurrency(double x) {
		if (Double.isNaN(x) || Double.isInfinite(x))
			return String.valueOf(x);
		return df.format(x).replaceAll(",", ".");
	}

	/** Calculates the unified statistic */
	public void calculate() {
		finalStat = new Statistic();

		//Iterate through all statistics
		Iterator<Statistic> sIt = stat.iterator();
		while (sIt.hasNext()) {
			Statistic sTemp = sIt.next();

			//Iterate through all weapon statistics
			Iterator<WeaponStatistic> wsIt = sTemp.weaponStats.iterator();
			while (wsIt.hasNext()) {
				WeaponStatistic wsTemp = wsIt.next();

				boolean added = false;

				//Test, if weapon already is added to the final weapon statistic
				Iterator<WeaponStatistic> fwsIt = finalStat.weaponStats.iterator();
				while (fwsIt.hasNext()) {
					WeaponStatistic fwsTemp = fwsIt.next();

					//If so...
					if (fwsTemp.weaponName.equals(wsTemp.weaponName)) {
						added = true;
						//Summarize weapon data
						fwsTemp.shots     += wsTemp.shots;
						fwsTemp.hits      += wsTemp.hits;
						fwsTemp.kills     += wsTemp.kills;
						fwsTemp.deaths    += wsTemp.deaths;
						fwsTemp.headshots += wsTemp.headshots;
						//Calculate accuracies
						fwsTemp.acc     = (fwsTemp.hits      * 100d) / fwsTemp.shots;
						fwsTemp.headAcc = (fwsTemp.headshots * 100d) / fwsTemp.hits;
						break;
					}
				}

				//If not, use current weapon statistic as a basis
				if (!added)
					finalStat.weaponStats.add(wsTemp);
			}
		}

		//Iterate through all statistics (again)
		sIt = stat.iterator();
		while (sIt.hasNext()) {
			Statistic sTemp = sIt.next();

			//Summarize all other field of the statistics
			finalStat.rounds             += sTemp.rounds;
			finalStat.damageGiven        += sTemp.damageGiven;
			finalStat.damageReceived     += sTemp.damageReceived;
			finalStat.teamDamageGiven    += sTemp.teamDamageGiven;
			finalStat.teamDamageReceived += sTemp.teamDamageReceived;
			finalStat.gibs               += sTemp.gibs;
			finalStat.kills              += sTemp.kills;
			finalStat.deaths             += sTemp.deaths;
			finalStat.eff                 = (double)finalStat.kills / finalStat.deaths;
		}
	}

	private static void writeln(BufferedWriter bw, String s) throws IOException {
		bw.write(s);
		bw.newLine();
	}

	/** Write a statistic to a stream using a custom format. */
	private static void output(BufferedWriter bw, Statistic stat) throws IOException {
		if (stat.map != null)
			writeln(bw, "Map: " + stat.map);
		if (stat.playerName != null)
			writeln(bw, "Playername: " + stat.playerName);
		writeln(bw, "Rounds: " + stat.rounds);
		bw.newLine();
		writeln(bw, "Damage Given: "    + stat.damageGiven);
		writeln(bw, "Damage Received: " + stat.damageReceived);
		writeln(bw, "Damage Quote: "    + roundCurrency((double)stat.damageGiven / stat.damageReceived));
		writeln(bw, "Team Damage Given: "    + stat.teamDamageGiven);
		writeln(bw, "Team Damage Received: " + stat.teamDamageReceived);
		writeln(bw, "Team Damage Quote: "    + roundCurrency((double)stat.teamDamageGiven / stat.teamDamageReceived));
		bw.newLine();
		writeln(bw, "Gibs: " + stat.gibs);
		if (stat.rank != null) {
			writeln(bw, "Rank: "   + stat.rank);
			writeln(bw, "XP: "     + stat.experience);
		}
		writeln(bw, "Kills: "      + stat.kills);
		writeln(bw, "Deaths: "     + stat.deaths);
		writeln(bw, "Efficiency: " + roundCurrency(stat.eff));
		bw.newLine();

		//Iterate trough the weapon statistics
		Iterator<WeaponStatistic> wsIt = stat.weaponStats.iterator();
		while (wsIt.hasNext()) {
			WeaponStatistic wStat = wsIt.next();

			writeln(bw, wStat.weaponName);
			writeln(bw, "accuracy "     + roundCurrency(wStat.acc) + "%");
			writeln(bw, "headshotrate " + roundCurrency(wStat.headAcc) + "%");
			writeln(bw, "shots / kill " + roundCurrency((double)wStat.shots / wStat.kills));
			writeln(bw, "hits / kill "  + roundCurrency((double)wStat.hits  / wStat.kills));
			writeln(bw, "shots "        + wStat.shots);
			writeln(bw, "hits "         + wStat.hits);
			writeln(bw, "kills "        + wStat.kills);
			writeln(bw, "deaths "       + wStat.deaths);
			writeln(bw, "headshots "    + wStat.headshots);
			bw.newLine();
		}

		bw.newLine();
		bw.flush();
	}

	/** Write a list of statistics to a stream using a custom format. */
	private static void output(BufferedWriter bw, ArrayList<Statistic> stats)
			throws IOException {
		Iterator<Statistic> sIt = stats.iterator();
		while (sIt.hasNext())
			output(bw, sIt.next());
	}

	/** Write the list of parsed statistics to a stream using a custom format. */
	public void outputStats(BufferedWriter bw) throws IOException {
		output(bw, stat);
	}

	/**
	 * Write the unified statistic to a stream using a custom format, if the
	 * unified statistic was calculated.
	 */
	public boolean outputFinalStat(BufferedWriter bw) throws IOException {
		if (finalStat != null) {
			output(bw, finalStat);
			return true;
		}
		return false;
	}

	/** Write the list of parsed statistics to a file using a custom format. */
	public void outputStats(File f) throws IOException {
		BufferedWriter bw = new BufferedWriter(new FileWriter(f));
		output(bw, stat);
		bw.close();
	}

	/**
	 * Write the unified statistic to a file using a custom format, if the
	 * unified statistic was calculated.
	 */
	public boolean outputFinalStat(File f) throws IOException {
		if (finalStat != null) {
			BufferedWriter bw = new BufferedWriter(new FileWriter(f));
			output(bw, finalStat);
			bw.close();
			return true;
		}
		return false;
	}
}