//
// FoXBot - AI Bot for Halflife's Team Fortress Classic
//
// (http://foxbot.net)
//
// waypoint.h
//
// Copyright (C) 2003 - Tom "Redfox" Simpson
//
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
//
// See the GNU General Public License for more details at:
// http://www.gnu.org/copyleft/gpl.html
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#ifndef WAYPOINT_H
#define WAYPOINT_H

#include <limits.h>

#define MAX_WAYPOINTS 32000

#define REACHABLE_RANGE 800.0

// defines for waypoint flags field (32 bits are available)
#define W_FL_TEAM			((1<<0) + (1<<1))  /* allow for 4 teams (0-3) */
#define W_FL_TEAM_SPECIFIC	(1<<2)  /* waypoint only for specified team */
#define W_FL_CROUCH			(1<<3)  /* must crouch to reach this waypoint */
#define W_FL_LADDER			(1<<4)  /* waypoint on a ladder */
#define W_FL_LIFT			(1<<5)  /* wait for lift to be down before approaching this waypoint */
#define W_FL_DOOR			(1<<6)  /* wait for door to open */
#define W_FL_HEALTH			(1<<7)  /* health kit (or wall mounted) location */
#define W_FL_ARMOR			(1<<8)  /* armor (or HEV) location */
#define W_FL_AMMO			(1<<9)  /* ammo location */
#define W_FL_SNIPER			(1<<10) /* sniper waypoint (a good sniper spot) */

#define W_FL_TFC_FLAG		(1<<11) /* flag position (or hostage or president) */
#define W_FL_FLF_CAP		(1<<11) /* Front Line Force capture point */

#define W_FL_TFC_FLAG_GOAL	(1<<12) /* flag return position (or rescue zone) */
#define W_FL_FLF_DEFEND		(1<<12) /* Front Line Force defend point */

#define W_FL_PRONE			(1<<13) /* go prone (laying down) */
#define W_FL_TFC_DEFEND		(1<<13) /* sentry gun */
#define W_FL_TFC_DEFEND180  (1<<28)  /* 180 SG point!! */
#define W_FL_AIMING			(1<<14) /* aiming waypoint */

#define W_FL_JUMP			(1<<15) /* jump waypoint */

#define W_FL_POINT1			(1<<16) /* point1 */
#define W_FL_POINT2			(1<<17) /* point2 */
#define W_FL_POINT3			(1<<18) /* point3 */
#define W_FL_POINT4			(1<<19) /* point4 */
#define W_FL_POINT5			(1<<20) /* point5 */
#define W_FL_POINT6			(1<<21) /* point6 */
#define W_FL_POINT7			(1<<22) /* point7 */
#define W_FL_POINT8			(1<<23) /* point8 */

#define W_FL_TFC_PIPETRAP   (1<<24) /* demoman pipetrap */
#define W_FL_TFC_DEMO		(1<<25) /* demoman detpack */

//#define W_FL_TFC_DISP		(1<<26) /* dispencer */
#define W_FL_TFC_JUMP		(1<<27)  /* rocket/conc jump */

#define W_FL_TFC_TELEPORTER_ENTRANCE	(1<<26)
#define W_FL_TFC_TELEPORTER_EXIT		(1<<30)

#define W_FL_TFC_PL_DEFEND  (1<<29)  /* bot defense point!! */

#define W_FL_DELETED		(1<<31) /* used by waypoint allocation code */

#define WAYPOINT_VERSION 4

// define the waypoint file header structure...
typedef struct {
   char filetype[8];  // should be "FoXBot\0"
   int  waypoint_file_version;
   int  waypoint_file_flags;  // not currently used
   int  number_of_waypoints;
   char mapname[32];  // name of map for these waypoints
} WAYPOINT_HDR;


// define the structure for waypoints...
typedef struct {
   int    flags;    // button, lift, flag, health, ammo, etc.
   Vector origin;   // location
} WAYPOINT;



#define WAYPOINT_UNREACHABLE   UINT_MAX
#define WAYPOINT_MAX_DISTANCE (UINT_MAX-1)

#define MAX_PATH_INDEX 4

// define the structure for waypoint paths (paths are connections between
// two waypoint nodes that indicates the bot can get from point A to point B.
// note that paths DON'T have to be two-way.  sometimes they are just one-way
// connections between two points.  There is an array called "paths" that
// contains head pointers to these structures for each waypoint index.
typedef struct path {
   short int index[MAX_PATH_INDEX];  // indexes of waypoints (index -1 means not used)
   struct path *next;   // link to next structure
} PATH;

#define A_FL_1 (1<<0)
#define A_FL_2 (1<<1)
#define A_FL_3 (1<<2)
#define A_FL_4 (1<<3)

typedef struct area {
   Vector a;   // location
   Vector b;   // location
   Vector c;   // location
   Vector d;   // location
   char namea[64]; //team1's name
   char nameb[64]; //team2..
   char namec[64];
   char named[64];
   int    flags;
} AREA;

#define AREA_VERSION 1

// define the waypoint file header structure...
typedef struct {
   char filetype[8];  // should be "FoXBot\0"
   int  area_file_version;
   int  number_of_areas;
   char mapname[32];  // name of map for these waypoints
} AREA_HDR;

// waypoint function prototypes...
void WaypointInit(void);
int  WaypointFindPath(PATH **pPath, int *path_index, int waypoint_index, int team);
int  WaypointFindNearest(edict_t *pEntity, float distance, int team);
int  WaypointFindNearest(Vector v_src, edict_t *pEntity, float range, int team);
int WaypointFindNearest(Vector v_src, float range, int team);
int  WaypointFindNearestGoal(edict_t *pEntity, int src, int team, int flags);
int  WaypointFindNearestGoal(Vector v_src, edict_t *pEntity, float range, int team, int flags);
int  WaypointFindRandomGoal(edict_t *pEntity, int team, int flags);
int  WaypointFindRandomGoal(Vector v_src, edict_t *pEntity, float range, int team, int flags);
int  WaypointFindNearestAiming(Vector v_origin);
void WaypointAdd(edict_t *pEntity);
void WaypointAddAiming(edict_t *pEntity);
void WaypointDelete(edict_t *pEntity);
void WaypointCreatePath(edict_t *pEntity, int cmd);
void WaypointRemovePath(edict_t *pEntity, int cmd);
bool WaypointLoad(edict_t *pEntity);
void WaypointSave(void);
bool WaypointReachable(Vector v_srv, Vector v_dest, edict_t *pEntity);
int  WaypointFindReachable(edict_t *pEntity, float range, int team);
void WaypointPrintInfo(edict_t *pEntity);
void WaypointThink(edict_t *pEntity);
void WaypointFloyds(int *shortest_path, int *from_to);
void WaypointRouteInit(void);
unsigned short WaypointRouteFromTo(int src, int dest, int team);
int  WaypointDistanceFromTo(int src, int dest, int team);
void WaypointDrawBeam(edict_t *pEntity, Vector start, Vector end, int width,
					  int noise, int red, int green, int blue, int brightness, int speed);
int WaypointMultiDistanceFromTo(bot_t *pBot,int src, int dest, int team);
bool WaypointAvailable(int index,int ii);
void WaypointRunOneWay(edict_t *pEntity);
void WaypointRunTwoWay(edict_t *pEntity);
void WaypointAutoBuild(edict_t *pEntity);
//was gonna cal paths before it started
//now it'll just do it on the fly
void AreaDefCreate(edict_t *pEntity);
int AreaDefPointFindNearest(edict_t *pEntity, float range, int flags);
void AreaDefDelete(edict_t *pEntity);
void AreaDefSave(void);
bool AreaDefLoad(edict_t *pEntity);
void AreaDefPrintInfo(edict_t *pEntity);
bool AreaInside(edict_t *pEntity,int i);
int AreaInsideClosest(edict_t *pEntity);
void AreaAutoBuild1(void);
void AreaAutoMerge(void);
void JunctionCalculate();
int JunctionGoto(bot_t *pBot,int from,int to);

void ProcessCommanderList(void);
#endif // WAYPOINT_H
