//
// FoXBot - AI Bot for Halflife's Team Fortress Classic
//
// (http://foxbot.net)
//
// waypoint.cpp
//
// Copyright (C) 2003 - Tom "Redfox" Simpson
//
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  
//
// See the GNU General Public License for more details at:
// http://www.gnu.org/copyleft/gpl.html
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

#ifndef __linux__
#  include <io.h>
#else
#  include <unistd.h>
#endif
#include <fcntl.h>

#ifndef __linux__
#  include <sys\stat.h>
#else
#  include <sys/stat.h>
#endif

#include "extdll.h"
#include <enginecallback.h>
#include "util.h"
//#include "cbase.h"

#include "bot.h"
#include "waypoint.h"

//me
#include "engine.h"

// linked list class.
#include "list.h"
List<char *> commanders;

int last_area=-1;
bool area_on_last;

// Rocket jumping points
// The array to hold all the RJ waypoints, and the team they belong to.
int RJPoints[bot_t::MAXRJWAYPOINTS][2];

extern int mod_id;
extern int m_spriteTexture;
extern bool botcamEnabled;

//my externs
extern bool blue_av[8];
extern bool red_av[8];
extern bool green_av[8];
extern bool yellow_av[8];

extern bool attack[4]; //teams attack
extern bool defend[4]; //teams defend

// waypoints with information bits (flags)
WAYPOINT waypoints[MAX_WAYPOINTS];

// number of waypoints currently in use
int num_waypoints;

//waypoint author
char waypoint_author[256];

// declare the array of head pointers to the path structures...
PATH *paths[MAX_WAYPOINTS];

// time that this waypoint was displayed (while editing)
float wp_display_time[MAX_WAYPOINTS];

bool g_waypoint_paths = FALSE;  // have any paths been allocated?
bool g_waypoint_cache = FALSE;
bool g_waypoint_on = FALSE;
bool g_auto_waypoint = FALSE;
bool g_path_waypoint = FALSE;
bool g_find_waypoint = FALSE;
long g_find_wp = 0;
bool g_path_connect = TRUE;
bool g_path_oneway = FALSE;
bool g_path_twoway = FALSE;
Vector last_waypoint;
float f_path_time = 0.0;

float a_display_time[MAX_WAYPOINTS];
bool g_area_def;

int wpt1;
int wpt2;

//area defs...
AREA areas[MAX_WAYPOINTS];
int num_areas;
bool is_junction[MAX_WAYPOINTS];

//my stuf
extern int drop_flag[32][4];
extern edict_t *drop_flage[32][4];

//multipath vars
//unsigned short int temp[1024*1024];
//bool check[1024];
//bool r_check[1024];
//int full_temp[1024];
//int cur_temp[1024]; //shortest route can only ever return max waypoints..
float err_count;
int ignore;
int ignore_lev;

unsigned int route_num_waypoints;
unsigned int *shortest_path[4] = {NULL, NULL, NULL, NULL};
unsigned int *from_to[4] = {NULL, NULL, NULL, NULL};

static FILE *fp;


void WaypointDebug(void)
{
	int y = 1, x = 1;

	fp=fopen("bot.txt","a");
	fprintf(fp,"WaypointDebug: LINKED LIST ERROR!!!\n");
	fclose(fp);

	x = x - 1;  // x is zero
	y = y / x;  // cause an divide by zero exception

	return;
}


// free the linked list of waypoint path nodes...
void WaypointFree(void)
{
	for (int i=0; i < MAX_WAYPOINTS; i++)
	{
		int count = 0;

		if (paths[i])
		{
			PATH *p = paths[i];
			PATH *p_next;

			while (p)  // free the linked list
			{
				p_next = p->next;  // save the link to next
				free(p);
				p = p_next;

#ifdef _DEBUG
				count++;
				if (count > 1000) WaypointDebug();
#endif
			}

			paths[i] = NULL;
		}
	}
}


// initialize the waypoint structures...
void WaypointInit(void)
{
	int i;

	// have any waypoint path nodes been allocated yet?
	if (g_waypoint_paths)
		WaypointFree();  // must free previously allocated path memory

	for (i=0; i < 4; i++)
	{
		if (shortest_path[i] != NULL)
			free(shortest_path[i]);

		if (from_to[i] != NULL)
			free(from_to[i]);
	}

	for(i=0;i<256;i++)
		waypoint_author[i]=NULL;

	for (i=0; i < MAX_WAYPOINTS; i++)
	{
		waypoints[i].flags = 0;
		waypoints[i].origin = Vector(0,0,0);

		wp_display_time[i] = 0.0;

		paths[i] = NULL;  // no paths allocated yet

		a_display_time[i] = 0.0;
		areas[i].flags = 0;
		areas[i].a = Vector(0,0,0);
		areas[i].b = Vector(0,0,0);
		areas[i].c = Vector(0,0,0);
		areas[i].d = Vector(0,0,0);
		areas[i].namea[0] = NULL;
		areas[i].nameb[0] = NULL;
		areas[i].namec[0] = NULL;
		areas[i].named[0] = NULL;

		is_junction[i]=false;
	}

	f_path_time = 0.0;  // reset waypoint path display time

	num_waypoints = 0;
	num_areas = 0;

	last_waypoint = Vector(0,0,0);

	for (i=0; i < 4; i++)
	{
		shortest_path[i] = NULL;
		from_to[i] = NULL;
	}
}


// add a path from one waypoint (add_index) to another (path_index)...
void WaypointAddPath(short int add_index, short int path_index)
{
	//don't do it if its greater then max distance
	if((waypoints[add_index].origin - waypoints[path_index].origin).Length() > REACHABLE_RANGE)
		return;
	PATH *p, *prev;
	int i;
	int count = 0;

	p = paths[add_index];
	prev = NULL;

	// find an empty slot for new path_index...
	while (p != NULL)
	{
		i = 0;

		while (i < MAX_PATH_INDEX)
		{
			//don't add the path if its already their?!
			if(p->index[i] == path_index)
				return;
			if (p->index[i] == -1)
			{
				p->index[i] = path_index;

				return;
			}

			i++;
		}

		prev = p;     // save the previous node in linked list
		p = p->next;  // go to next node in linked list

#ifdef _DEBUG
		count++;
		if (count > 100) WaypointDebug();
#endif
	}

	//hmmm, should we only do this if prev->next !=NULL
	// nope loop only stops when complete.. ignore this unless future probs arrise!

	p = (PATH *)malloc(sizeof(PATH));

	if (p == NULL)
	{
		ALERT(at_error, "FoXBot - Error allocating memory for path!");
	}

	p->index[0] = path_index;
	p->index[1] = -1;
	p->index[2] = -1;
	p->index[3] = -1;
	p->next = NULL;

	if (prev != NULL)
		prev->next = p;  // link new node into existing list

	if (paths[add_index] == NULL)
		paths[add_index] = p;  // save head point if necessary
}


// delete all paths to this waypoint index...
void WaypointDeletePath(short int del_index)
{
	PATH *p;
	int index, i;

	// search all paths for this index...
	for (index=0; index < num_waypoints; index++)
	{
		p = paths[index];

		int count = 0;

		// search linked list for del_index...
		while (p != NULL)
		{
			i = 0;

			while (i < MAX_PATH_INDEX)
			{
				if (p->index[i] == del_index)
				{
					p->index[i] = -1;  // unassign this path
				}

				i++;
			}

			p = p->next;  // go to next node in linked list

#ifdef _DEBUG
			count++;
			if (count > 100) WaypointDebug();
#endif
		}
	}
}


// delete a path from a waypoint (path_index) to another waypoint
// (del_index)...
void WaypointDeletePath(short int path_index, short int del_index)
{
	PATH *p;
	int i;
	int count = 0;

	p = paths[path_index];

	// search linked list for del_index...
	while (p != NULL)
	{
		i = 0;

		while (i < MAX_PATH_INDEX)
		{
			if (p->index[i] == del_index)
			{
				p->index[i] = -1;  // unassign this path
			}

			i++;
		}

		p = p->next;  // go to next node in linked list

#ifdef _DEBUG
		count++;
		if (count > 100) WaypointDebug();
#endif
	}
}

// find a path from the current waypoint. (pPath MUST be NULL on the
// initial call. subsequent calls will return other paths if they exist.)
int WaypointFindPath(PATH **pPath, int *path_index, int waypoint_index, int team)
{
	int index;
	int count = 0;

	if (*pPath == NULL)
	{
		*pPath = paths[waypoint_index];
		*path_index = 0;
	}

	if (*path_index == MAX_PATH_INDEX)
	{
		*path_index = 0;

		*pPath = (*pPath)->next;  // go to next node in linked list
	}

	while (*pPath != NULL)
	{
		while (*path_index < MAX_PATH_INDEX)
		{
			if ((*pPath)->index[*path_index] != -1)  // found a path?
			{
				// save the return value
				index = (*pPath)->index[*path_index];

				// skip this path if next waypoint is team specific and NOT this team
				if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
					((waypoints[index].flags & W_FL_TEAM) != team))
				{
					(*path_index)++;
					continue;
				}

				// set up stuff for subsequent calls...
				(*path_index)++;

				return index;
			}

			(*path_index)++;
		}

		*path_index = 0;

		*pPath = (*pPath)->next;  // go to next node in linked list

#ifdef _DEBUG
		count++;
		if (count > 100) WaypointDebug();
#endif
	}

	return -1;
}


// find the nearest waypoint to the player and return the index
// (-1 if not found)...
int WaypointFindNearest(edict_t *pEntity, float range, int team)
{
	int i, min_index;
	float distance;
	float min_distance;
	TraceResult tr;

	if (num_waypoints < 1)
		return -1;

	// find the nearest waypoint...

	min_index = -1;
	min_distance = 9999.0;

	for (i=0; i < num_waypoints; i++)
	{
		if (waypoints[i].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[i].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[i].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[i].flags & W_FL_TEAM) != team))
			continue;	  

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[i].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[i].origin, 300))
			continue;

		distance = (waypoints[i].origin - pEntity->v.origin).Length();

		if ((distance < min_distance) && (distance < range))
		{
			// if waypoint is visible from current position (even behind head)...
			UTIL_TraceLine( pEntity->v.origin + pEntity->v.view_ofs, waypoints[i].origin,
				ignore_monsters, pEntity->v.pContainingEntity, &tr );

			if (tr.flFraction >= 1.0)
			{
				min_index = i;
				min_distance = distance;
			}
		}
	}

	return min_index;
}


// find the nearest waypoint to the source postition and return the index
// of that waypoint...
int WaypointFindNearest(Vector v_src, edict_t *pEntity, float range, int team)
{
	int index, min_index;
	float distance;
	float min_distance;
	TraceResult tr;

	if (num_waypoints < 1)
		return -1;

	// find the nearest waypoint...

	min_index = -1;
	min_distance = 9999.0;

	for (index=0; index < num_waypoints; index++)
	{
		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;

		distance = (waypoints[index].origin - v_src).Length();

		if ((distance < min_distance) && (distance < range))
		{
			// if waypoint is visible from source position...
			UTIL_TraceLine( v_src, waypoints[index].origin, ignore_monsters,
				pEntity->v.pContainingEntity, &tr );

			if (tr.flFraction >= 1.0)
			{
				min_index = index;
				min_distance = distance;
			}
		}
	}

	return min_index;
}
//pic a origin, and find waypoints near it
int WaypointFindNearest(Vector v_src, float range, int team)
{
	int index, min_index;
	float distance;
	float min_distance;
	TraceResult tr;

	if (num_waypoints < 1)
		return -1;

	// find the nearest waypoint...

	min_index = -1;
	min_distance = 9999.0;

	for (index=0; index < num_waypoints; index++)
	{
		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;

		distance = (waypoints[index].origin - v_src).Length();

		if ((distance < min_distance) && (distance < range))
		{
			min_index = index;
			min_distance = distance;
		}
	}
	return min_index;
}

// find the goal nearest to the player matching the "flags" bits and return
// the index of that waypoint...
int WaypointFindNearestGoal(edict_t *pEntity, int src, int team, int flags)
{
	int index, min_index;
	int distance, min_distance;

	if (num_waypoints < 1)
		return -1;

	// find the nearest waypoint with the matching flags...

	min_index = -1;
	min_distance = 99999;

	for (index=0; index < num_waypoints; index++)
	{
		if ((waypoints[index].flags & flags) != flags)
			continue;  // skip this waypoint if the flags don't match

		if (index == src)
			continue;  // skip the source waypoint

		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		// Skip unavailable waypoints.
		if(!WaypointAvailable(index, team))
			continue; 

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;
		
		distance = WaypointDistanceFromTo(src, index, team);

		if (distance < min_distance && distance>0)
		{
			min_index = index;
			min_distance = distance;
		}
	}

	return min_index;
}


// find the goal nearest to the source position (v_src) matching the "flags"
// bits and return the index of that waypoint...
int WaypointFindNearestGoal(Vector v_src, edict_t *pEntity, float range, int team, int flags)
{
	int index, min_index;
	int distance, min_distance;

	if (num_waypoints < 1)
		return -1;

	// find the nearest waypoint with the matching flags...

	min_index = -1;
	min_distance = 99999;

	for (index=0; index < num_waypoints; index++)
	{
		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		// Skip unavailable waypoints.
		if(!WaypointAvailable(index, team))
			continue; 

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;

		if ((waypoints[index].flags & flags) != flags)
			continue;  // skip this waypoint if the flags don't match

		distance = (waypoints[index].origin - v_src).Length();

		if ((distance < range) && (distance < min_distance))
		{
			min_index = index;
			min_distance = distance;
		}
	}

	return min_index;
}


// find a random goal matching the "flags" bits and return the index of
// that waypoint...
int WaypointFindRandomGoal(edict_t *pEntity, int team, int flags)
{
	int index;
	int indexes[50];
	int count = 0;

	if (num_waypoints < 1)
		return -1;

	//ok, slip in code here that check for the dropped flags
	//i.e. should we clear the data if their no longer at that wpt
	// /*
	for(int ind=0;ind<4;ind++)
	{
		count=0;
		while(count<32 && num_waypoints>0)
		{
			//{ fp=fopen("bot.txt", "a"); fprintf(fp, "team %d e %x d %d n %d count %d ind %d\n",team,pEntity,drop_flag[count][ind],num_waypoints,count,ind); fclose(fp); }
			if(drop_flag[count][ind]>num_waypoints)
				drop_flag[count][ind]=-1;
			if(drop_flag[count][ind]<-1)
				drop_flag[count][ind]=-1;
			if(drop_flag[count][ind]!=-1)
			{
				//ok, data says we have a flag here
				//so does it exist?
				bool f=false;
				edict_t *pent;
				pent = NULL; 
				while ((pent = FIND_ENTITY_BY_CLASSNAME( pent, "item_tfgoal" )) != NULL &&
					(!FNullEnt(pent))) 
				{  
					float di=(pent->v.origin - waypoints[drop_flag[count][ind]].origin).Length();
					if(di<=200 && di!=0 && pent->v.model!=0 && drop_flage[count][ind]==pent)
					{
						f=true;
						break;
					}
				}
				if(!f)
				{
					pent=NULL;
					while ((pent = FIND_ENTITY_BY_CLASSNAME( pent, "info_tfgoal" )) != NULL &&
						(!FNullEnt(pent))) 
					{  
						float di=(pent->v.origin - waypoints[drop_flag[count][ind]].origin).Length();
						if(di<=200 && di!=0 && pent->v.model!=0 && drop_flage[count][ind]==pent)
						{
							f=true;
							break;
						}
					}
				}
				//clear the data, the flags no longer their
				if(!f) 
				{
					drop_flag[count][ind]=-1;
					drop_flage[count][ind]=NULL;
				}
			}
			count++;
		}

	}

	count=0;
	//we will always go for dropped falgs..
	//is this a good idea??
	//no, put a catch in, if count>1 then % chance of not going for dropped
	if(flags==W_FL_TFC_FLAG && team!=-1 && mod_id==TFC_DLL)
	{
		index=0;
		while(index<32)
		{
			if (count < 50 && drop_flag[index][team]!=-1)
			{
				indexes[count] = drop_flag[index][team];
				count++;
			}
			index++;
		}

		if (count != 0) // found atleast 1 dropped flag
		{
			index = RANDOM_LONG(1, count) - 1;
			//so only go for dropped flags, if theirs only 1
			if(count==1) return indexes[index];
			//otherwise?! may go for this dropped flag
			//hmm, dunno of any maps with multiple dropped flags for a specific team..hmm
			//but anyways, this case should be used
			//better to have it I think.
			else if(RANDOM_LONG(0,100)<80) return indexes[index];
		}
	} // */
	count=0;
	// find all the waypoints with the matching flags...

	for (index=0; index < num_waypoints; index++)
	{
		// DrEvils rocketjump pathing
		if ((waypoints[index].flags & W_FL_TFC_JUMP) && (pEntity->v.playerclass != TFC_CLASS_SOLDIER))
			continue;

		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		if ((waypoints[index].flags & flags) != flags)
			continue;  // skip this waypoint if the flags don't match

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;

		//need to skip wpt if not available to bots team
		if(UTIL_GetBotIndex(pEntity)!=-1)
		{
			int i,j = -1;
			if((waypoints[index].flags & W_FL_POINT1) == W_FL_POINT1) j=0;
			if((waypoints[index].flags & W_FL_POINT2) == W_FL_POINT2) j=1;
			if((waypoints[index].flags & W_FL_POINT3) == W_FL_POINT3) j=2;
			if((waypoints[index].flags & W_FL_POINT4) == W_FL_POINT4) j=3;
			if((waypoints[index].flags & W_FL_POINT5) == W_FL_POINT5) j=4;
			if((waypoints[index].flags & W_FL_POINT6) == W_FL_POINT6) j=5;
			if((waypoints[index].flags & W_FL_POINT7) == W_FL_POINT7) j=6;
			if((waypoints[index].flags & W_FL_POINT8) == W_FL_POINT8) j=7;
			if(j!=-1)
			{
				i=UTIL_GetTeam(pEntity);
				if(i==0 && blue_av[j]==false) continue;
				if(i==1 && red_av[j]==false) continue;
				if(i==2 && yellow_av[j]==false) continue;
				if(i==3 && green_av[j]==false) continue;
			}
		}

		if (count < 50)
		{
			indexes[count] = index;

			count++;
		}
	}

	if (count == 0)  // no matching waypoints found
		return -1;

	index = RANDOM_LONG(1, count) - 1;
	return indexes[index];
}


// find a random goal within a range of a position (v_src) matching the
// "flags" bits and return the index of that waypoint...
int WaypointFindRandomGoal(Vector v_src, edict_t *pEntity, float range, int team, int flags)
{
	int index;
	int indexes[50];
	int count = 0;
	float distance;

	if (num_waypoints < 1)
		return -1;

	// find all the waypoints with the matching flags...

	for (index=0; index < num_waypoints; index++)
	{
		// DrEvils rocketjump pathing
		if ((waypoints[index].flags & W_FL_TFC_JUMP) && (pEntity->v.playerclass != TFC_CLASS_SOLDIER))
			continue;

		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[index].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[index].flags & W_FL_TEAM) != team))
			continue;

		// DrEvil : skip points that already have an sg if I'm an engy
		if ((waypoints[index].flags & W_FL_TFC_DEFEND) && (pEntity->v.playerclass == TFC_CLASS_ENGINEER) && BotSGAtPoint(waypoints[index].origin, 300))
			continue;

		if ((waypoints[index].flags & flags) != flags)
			continue;  // skip this waypoint if the flags don't match

		distance = (waypoints[index].origin - v_src).Length();

		if ((distance < range) && (count < 50))
		{
			indexes[count] = index;

			count++;
		}
	}

	if (count == 0)  // no matching waypoints found
		return -1;

	index = RANDOM_LONG(1, count) - 1;

	return indexes[index];
}


// find the nearest "special" aiming waypoint (for sniper aiming)...
int WaypointFindNearestAiming(Vector v_origin)
{
	int index;
	int min_index = -1;
	int min_distance = 9999.0;
	float distance;

	if (num_waypoints < 1)
		return -1;

	// search for nearby aiming waypoint...
	for (index=0; index < num_waypoints; index++)
	{
		if (waypoints[index].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if ((waypoints[index].flags & W_FL_AIMING) == 0)
			continue;  // skip any NON aiming waypoints

		distance = (v_origin - waypoints[index].origin).Length();

		if ((distance < min_distance) && (distance < 200))
		{
			min_index = index;
			min_distance = distance;
		}
	}

	return min_index;
}


void WaypointDrawBeam(edict_t *pEntity, Vector start, Vector end, int width,
					  int noise, int red, int green, int blue, int brightness, int speed)
{
	MESSAGE_BEGIN(MSG_ONE, SVC_TEMPENTITY, NULL, pEntity);
	WRITE_BYTE( TE_BEAMPOINTS);
	WRITE_COORD(start.x);
	WRITE_COORD(start.y);
	WRITE_COORD(start.z);
	WRITE_COORD(end.x);
	WRITE_COORD(end.y);
	WRITE_COORD(end.z);
	WRITE_SHORT( m_spriteTexture );
	WRITE_BYTE( 1 ); // framestart
	WRITE_BYTE( 10 ); // framerate
	WRITE_BYTE( 10 ); // life in 0.1's
	WRITE_BYTE( width ); // width
	WRITE_BYTE( noise );  // noise

	WRITE_BYTE( red );   // r, g, b
	WRITE_BYTE( green );   // r, g, b
	WRITE_BYTE( blue );   // r, g, b

	WRITE_BYTE( brightness );   // brightness
	WRITE_BYTE( speed );    // speed
	MESSAGE_END();
}


void WaypointAdd(edict_t *pEntity)
{
	int index;
	edict_t *pent = NULL;
	float radius = 40;
	char item_name[64];

	if (num_waypoints >= MAX_WAYPOINTS)
		return;

	index = 0;

	// find the next available slot for the new waypoint...
	while (index < num_waypoints)
	{
		if ((waypoints[index].flags & W_FL_DELETED)==W_FL_DELETED)
		{
			break;
		}

		index++;
	}

	waypoints[index].flags = 0;

	// store the origin (location) of this waypoint (use entity origin)
	waypoints[index].origin = pEntity->v.origin;

	// store the last used waypoint for the auto waypoint code...
	last_waypoint = pEntity->v.origin;

	// set the time that this waypoint was originally displayed...
	wp_display_time[index] = gpGlobals->time;


	Vector start, end;

	start = pEntity->v.origin - Vector(0, 0, 34);
	end = start + Vector(0, 0, 68);

	if ((pEntity->v.flags & FL_DUCKING) == FL_DUCKING)
	{
		waypoints[index].flags |= W_FL_CROUCH;  // crouching waypoint

		start = pEntity->v.origin - Vector(0, 0, 17);
		end = start + Vector(0, 0, 34);
	}

	if (pEntity->v.movetype == MOVETYPE_FLY)
		waypoints[index].flags |= W_FL_LADDER;  // waypoint on a ladder


	//********************************************************
	// look for lift, ammo, flag, health, armor, etc.
	//********************************************************

	while ((pent = FIND_ENTITY_IN_SPHERE( pent, pEntity->v.origin, radius )) != NULL &&
		(!FNullEnt(pent)))
	{
		strcpy(item_name, STRING(pent->v.classname));

		if (strcmp("item_healthkit", item_name) == 0)
		{
			ClientPrint(pEntity, HUD_PRINTCONSOLE, "found a healthkit!\n");
			waypoints[index].flags |= W_FL_HEALTH;
		}

		if (strncmp("item_armor", item_name, 10) == 0)
		{
			ClientPrint(pEntity, HUD_PRINTCONSOLE, "found some armor!\n");
			waypoints[index].flags |= W_FL_ARMOR;
		}

		// *************
		// LOOK FOR AMMO
		// *************

	}

	// draw a blue waypoint
	WaypointDrawBeam(pEntity, start, end, 30, 0, 0, 0, 255, 250, 5);

	EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
		ATTN_NORM, 0, 100);

	// increment total number of waypoints if adding at end of array...
	if (index == num_waypoints)
		num_waypoints++;

	//delete stuff just incase.
	WaypointDeletePath(index);
	//delete from index to other..cus his delete function dont!

	PATH *p;
	int  i;
	p = paths[index];
	int count = 0;
	// search linked list for del_index...
	while (p != NULL)
	{
		i = 0;
		while (i < MAX_PATH_INDEX)
		{
			p->index[i] = -1;  // unassign this path
			i++;
		}
		p = p->next;  // go to next node in linked list

#ifdef _DEBUG
		count++;
		if (count > 100) WaypointDebug();
#endif
	}

	if(g_path_connect)
	{
		// calculate all the paths to this new waypoint
		for (int i=0; i < num_waypoints; i++)
		{
			if (i == index)
				continue;  // skip the waypoint that was just added

			if (waypoints[i].flags & W_FL_AIMING)
				continue;  // skip any aiming waypoints

			if ((waypoints[i].flags & W_FL_DELETED)==W_FL_DELETED)
				continue; //skip deleted wpts!!

			// check if the waypoint is reachable from the new one (one-way)
			if ( WaypointReachable(pEntity->v.origin, waypoints[i].origin, pEntity) )
			{
				WaypointAddPath(index, i);
			}

			// check if the new one is reachable from the waypoint (other way)
			if ( WaypointReachable(waypoints[i].origin, pEntity->v.origin, pEntity) )
			{
				WaypointAddPath(i, index);
			}
		}
	}
}


void WaypointAddAiming(edict_t *pEntity)
{
	int index;
	edict_t *pent = NULL;

	if (num_waypoints >= MAX_WAYPOINTS)
		return;

	index = 0;

	// find the next available slot for the new waypoint...
	while (index < num_waypoints)
	{
		if (waypoints[index].flags & W_FL_DELETED)
			break;

		index++;
	}

	waypoints[index].flags = W_FL_AIMING;  // aiming waypoint

	Vector v_angle = pEntity->v.v_angle;

	v_angle.x = 0;  // reset pitch to horizontal
	v_angle.z = 0;  // reset roll to level

	UTIL_MakeVectors(v_angle);

	// store the origin (location) of this waypoint (use entity origin)
	waypoints[index].origin = pEntity->v.origin + gpGlobals->v_forward * 25;

	// set the time that this waypoint was originally displayed...
	wp_display_time[index] = gpGlobals->time;


	Vector start, end;

	start = pEntity->v.origin - Vector(0, 0, 10);
	end = start + Vector(0, 0, 14);

	// draw a blue waypoint
	WaypointDrawBeam(pEntity, start, end, 30, 0, 0, 0, 255, 250, 5);

	EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
		ATTN_NORM, 0, 100);

	// increment total number of waypoints if adding at end of array...
	if (index == num_waypoints)
		num_waypoints++;
}


void WaypointDelete(edict_t *pEntity)
{
	int index;
	int count = 0;

	if (num_waypoints < 1)
		return;

	index = WaypointFindNearest(pEntity, 50.0, -1);

	if (index == -1)
		return;
//update by yuraj
	if ((waypoints[index].flags & W_FL_SNIPER)== W_FL_SNIPER || (waypoints[index].flags & W_FL_TFC_DEFEND)== W_FL_TFC_DEFEND || (waypoints[index].flags & W_FL_TFC_TELEPORTER_ENTRANCE)== W_FL_TFC_TELEPORTER_ENTRANCE || (waypoints[index].flags & W_FL_TFC_TELEPORTER_EXIT)== W_FL_TFC_TELEPORTER_EXIT || (waypoints[index].flags & W_FL_TFC_PL_DEFEND)== W_FL_TFC_PL_DEFEND || (waypoints[index].flags & W_FL_TFC_PIPETRAP)== W_FL_TFC_PIPETRAP)
	{
		int i;
		int min_index = -1;
		int min_distance = 9999.0;
		float distance;

		// search for nearby aiming waypoint and delete it also...
		for (i=0; i < num_waypoints; i++)
		{
			if (waypoints[i].flags & W_FL_DELETED)
				continue;  // skip any deleted waypoints

			if ((waypoints[i].flags & W_FL_AIMING) == 0)
            continue;
			distance = (waypoints[i].origin - waypoints[index].origin).Length();

			if ((distance < min_distance) && (distance < 40))
			{
				min_index = i;
				min_distance = distance;
			}
		}

		if (min_index != -1)
		{
			waypoints[min_index].flags = W_FL_DELETED;  // not being used
			waypoints[min_index].origin = Vector(0,0,0);

			wp_display_time[min_index] = 0.0;
		}
	}

	// delete any paths that lead to this index...
	WaypointDeletePath(index);

	// free the path for this index...

	if (paths[index] != NULL)
	{
		PATH *p = paths[index];
		PATH *p_next;

		while (p)  // free the linked list
		{
			p_next = p->next;  // save the link to next
			free(p);
			p = p_next;

#ifdef _DEBUG
			count++;
			if (count > 100) WaypointDebug();
#endif
		}

		paths[index] = NULL;
	}

	waypoints[index].flags = W_FL_DELETED;  // not being used
	waypoints[index].origin = Vector(0,0,0);

	wp_display_time[index] = 0.0;

	EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/mine_activate.wav", 1.0,
		ATTN_NORM, 0, 100);
}


// allow player to manually create a path from one waypoint to another
void WaypointCreatePath(edict_t *pEntity, int cmd)
{
	static int waypoint1 = -1;  // initialized to unassigned
	static int waypoint2 = -1;  // initialized to unassigned

	if (cmd == 1)  // assign source of path
	{
		waypoint1 = WaypointFindNearest(pEntity, 50.0, -1);

		if (waypoint1 == -1)
		{
			// play "cancelled" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_moveselect.wav", 1.0,
				ATTN_NORM, 0, 100);

			return;
		}

		// play "start" sound...
		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_hudoff.wav", 1.0,
			ATTN_NORM, 0, 100);

		return;
	}

	if (cmd == 2)  // assign dest of path and make path
	{
		waypoint2 = WaypointFindNearest(pEntity, 50.0, -1);

		if ((waypoint1 == -1) || (waypoint2 == -1))
		{
			// play "error" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_denyselect.wav", 1.0,
				ATTN_NORM, 0, 100);

			return;
		}

		WaypointAddPath(waypoint1, waypoint2);

		// play "done" sound...
		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_hudon.wav", 1.0,
			ATTN_NORM, 0, 100);
	}
}


// allow player to manually remove a path from one waypoint to another
void WaypointRemovePath(edict_t *pEntity, int cmd)
{
	static int waypoint1 = -1;  // initialized to unassigned
	static int waypoint2 = -1;  // initialized to unassigned

	if (cmd == 1)  // assign source of path
	{
		waypoint1 = WaypointFindNearest(pEntity, 50.0, -1);

		if (waypoint1 == -1)
		{
			// play "cancelled" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_moveselect.wav", 1.0,
				ATTN_NORM, 0, 100);

			return;
		}

		// play "start" sound...
		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_hudoff.wav", 1.0,
			ATTN_NORM, 0, 100);

		return;
	}

	if (cmd == 2)  // assign dest of path and make path
	{
		waypoint2 = WaypointFindNearest(pEntity, 50.0, -1);

		if ((waypoint1 == -1) || (waypoint2 == -1))
		{
			// play "error" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_denyselect.wav", 1.0,
				ATTN_NORM, 0, 100);

			return;
		}

		WaypointDeletePath(waypoint1, waypoint2);

		// play "done" sound...
		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_hudon.wav", 1.0,
			ATTN_NORM, 0, 100);
	}
}


bool WaypointLoad(edict_t *pEntity)
{
	//{ fp=fopen("bot.txt","a"); fprintf(fp,"-waypoint load!\n"); fclose(fp); }
	char mapname[64];
	char filename[256];
	WAYPOINT_HDR header;
	char msg[80];
	int index, i;
	short int num;
	short int path_index;

	ProcessCommanderList();

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".fwp");

	FILE *bfp;

	UTIL_BuildFileName(filename, "waypoints", mapname);
	bfp=fopen(filename, "rb");

	if(bfp==NULL)
	{
		UTIL_BuildFileNameOld(filename, "maps", mapname);
		bfp=fopen(filename, "rb");
	}

	// if file exists, read the waypoint structure from it
	if (bfp != NULL)
	{
		if (IS_DEDICATED_SERVER())
			printf("loading waypoint file: %s\n", filename);
		fread(&header, sizeof(header), 1, bfp);

		header.filetype[7] = 0;
		if (strcmp(header.filetype, "FoXBot") == 0)
		{
			if (header.waypoint_file_version != WAYPOINT_VERSION)
			{
				if (pEntity)
					ClientPrint(pEntity, HUD_PRINTNOTIFY, "Incompatible FoXBot waypoint file version!\nWaypoints not loaded!\n");

				fclose(bfp);
				return FALSE;
			}

			header.mapname[31] = 0;

			if (stricmp(header.mapname, STRING(gpGlobals->mapname)) == 0)
			{
				WaypointInit();  // remove any existing waypoints

				for (i=0; i < header.number_of_waypoints; i++)
				{
					fread(&waypoints[i], sizeof(waypoints[0]), 1, bfp);
					num_waypoints++;
				}

				// read and add waypoint paths...
				for (index=0; index < num_waypoints; index++)
				{
					// read the number of paths from this node...
					fread(&num, sizeof(num), 1, bfp);

					for (i=0; i < num; i++)
					{
						fread(&path_index, sizeof(path_index), 1, bfp);

						WaypointAddPath(index, path_index);
					}
				}

				g_waypoint_paths = TRUE;  // keep track so path can be freed
				//read wpt name..if their is one (could be an old wpt file)
				fread(&waypoint_author, sizeof(char), 255, bfp);
			}
			else
			{
				if (pEntity)
				{
					sprintf(msg, "%s FoXBot waypoints are not for this map!\n", filename);
					ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
				}

				fclose(bfp);
				return FALSE;
			}
		}
		else
		{
			if (pEntity)
			{
				sprintf(msg, "%s is not a FoXBot waypoint file!\n", filename);
				ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
			}

			fclose(bfp);
			return FALSE;
		}

		fclose(bfp);

		WaypointRouteInit();
		return TRUE;
	}
	else
	{
		if (pEntity)
		{
			sprintf(msg, "Waypoint file %s does not exist!\nLooking for HPB bot file instead\n", filename);
			ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		}

		if (IS_DEDICATED_SERVER())
			printf("waypoint file %s not found!\nLooking for HPB bot file instead\n", filename);

	}


	//try for hpb_bot file instead

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".wpt");

	bfp=NULL;
	UTIL_BuildFileName(filename, "waypoints", mapname);
	bfp = fopen(filename, "rb");

	if(bfp==NULL)
	{
		UTIL_BuildFileNameOld(filename, "maps", mapname);
		bfp = fopen(filename, "rb");
	}

	// if file exists, read the waypoint structure from it
	if (bfp != NULL)
	{
		if (IS_DEDICATED_SERVER())
			printf("loading waypoint file: %s\n", filename);
		fread(&header, sizeof(header), 1, bfp);

		header.filetype[7] = 0;
		if (strcmp(header.filetype, "HPB_bot") == 0)
		{
			if (header.waypoint_file_version != WAYPOINT_VERSION)
			{
				if (pEntity)
					ClientPrint(pEntity, HUD_PRINTNOTIFY, "Incompatible HPB bot waypoint file version!\nWaypoints not loaded!\n");

				fclose(bfp);
				return FALSE;
			}

			header.mapname[31] = 0;

			if (strcmp(header.mapname, STRING(gpGlobals->mapname)) == 0)
			{
				WaypointInit();  // remove any existing waypoints

				for (i=0; i < header.number_of_waypoints; i++)
				{
					fread(&waypoints[i], sizeof(waypoints[0]), 1, bfp);
					num_waypoints++;
				}

				// read and add waypoint paths...
				for (index=0; index < num_waypoints; index++)
				{
					// read the number of paths from this node...
					fread(&num, sizeof(num), 1, bfp);

					for (i=0; i < num; i++)
					{
						fread(&path_index, sizeof(path_index), 1, bfp);

						WaypointAddPath(index, path_index);
					}
				}

				g_waypoint_paths = TRUE;  // keep track so path can be freed
			}
			else
			{
				if (pEntity)
				{
					sprintf(msg, "%s HPB bot waypoints are not for this map!\n", filename);
					ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
				}

				fclose(bfp);
				return FALSE;
			}
		}
		else
		{
			if (pEntity)
			{
				sprintf(msg, "%s is not a HPB bot waypoint file!\n", filename);
				ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
			}

			fclose(bfp);
			return FALSE;
		}

		fclose(bfp);

		WaypointSave();
		WaypointRouteInit();
	}
	else
	{
		if (pEntity)
		{
			sprintf(msg, "Waypoint file %s does not exist!\n", filename);
			ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		}

		if (IS_DEDICATED_SERVER())
			printf("waypoint file %s not found!\n", filename);

		return FALSE;
	}

	return TRUE;
}


void WaypointSave(void)
{
	char filename[256];
	char mapname[64];
	WAYPOINT_HDR header;
	int index, i;
	short int num;
	PATH *p;

	strcpy(header.filetype, "FoXBot");

	header.waypoint_file_version = WAYPOINT_VERSION;

	header.waypoint_file_flags = 0;  // not currently used

	header.number_of_waypoints = num_waypoints;

	memset(header.mapname, 0, sizeof(header.mapname));
	strncpy(header.mapname, STRING(gpGlobals->mapname), 31);
	header.mapname[31] = 0;

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".fwp");

	UTIL_BuildFileName(filename, "waypoints", mapname);

	FILE *bfp = fopen(filename, "wb");

	// write the waypoint header to the file...
	fwrite(&header, sizeof(header), 1, bfp);

	// write the waypoint data to the file...
	for (index=0; index < num_waypoints; index++)
	{
		fwrite(&waypoints[index], sizeof(waypoints[0]), 1, bfp);
	}

	// save the waypoint paths...
	for (index=0; index < num_waypoints; index++)
	{
		// count the number of paths from this node...

		p = paths[index];
		num = 0;

		while (p != NULL)
		{
			i = 0;

			while (i < MAX_PATH_INDEX)
			{
				if (p->index[i] != -1)
					num++;  // count path node if it's used

				i++;
			}

			p = p->next;  // go to next node in linked list
		}

		fwrite(&num, sizeof(num), 1, bfp);  // write the count

		// now write out each path index...

		p = paths[index];

		while (p != NULL)
		{
			i = 0;

			while (i < MAX_PATH_INDEX)
			{
				if (p->index[i] != -1)  // save path node if it's used
					fwrite(&p->index[i], sizeof(p->index[0]), 1, bfp);

				i++;
			}

			p = p->next;  // go to next node in linked list
		}
	}

	//write wpt name..if their is one (could be an old wpt file)
	fwrite(&waypoint_author, sizeof(char), 255, bfp);

	fclose(bfp);
}


bool WaypointReachable(Vector v_src, Vector v_dest, edict_t *pEntity)
{
	TraceResult tr;
	float curr_height, last_height;

	float distance = (v_dest - v_src).Length();

	// is the destination close enough?
	if (distance < REACHABLE_RANGE)
	{
		// check if this waypoint is "visible"...

		UTIL_TraceLine( v_src, v_dest, ignore_monsters,
			pEntity->v.pContainingEntity, &tr );

		// if waypoint is visible from current position (even behind head)...
		if (tr.flFraction >= 1.0)
		{
			// check for special case of both waypoints being underwater...
			if ((POINT_CONTENTS( v_src ) == CONTENTS_WATER) &&
				(POINT_CONTENTS( v_dest ) == CONTENTS_WATER))
			{
				return TRUE;
			}

			// check for special case of waypoint being suspended in mid-air...

			// is dest waypoint higher than src? (45 is max jump height)
			if (v_dest.z > (v_src.z + 45.0))
			{
				Vector v_new_src = v_dest;
				Vector v_new_dest = v_dest;

				v_new_dest.z = v_new_dest.z - 50;  // straight down 50 units

				UTIL_TraceLine(v_new_src, v_new_dest, dont_ignore_monsters,
					pEntity->v.pContainingEntity, &tr);

				// check if we didn't hit anything, if not then it's in mid-air
				if (tr.flFraction >= 1.0)
				{
					return FALSE;  // can't reach this one
				}
			}

			// check if distance to ground increases more than jump height
			// at points between source and destination...

			Vector v_direction = (v_dest - v_src).Normalize();  // 1 unit long
			Vector v_check = v_src;
			Vector v_down = v_src;

			v_down.z = v_down.z - 1000.0;  // straight down 1000 units

			UTIL_TraceLine(v_check, v_down, ignore_monsters,
				pEntity->v.pContainingEntity, &tr);

			last_height = tr.flFraction * 1000.0;  // height from ground

			distance = (v_dest - v_check).Length();  // distance from goal

			while (distance > 10.0)
			{
				// move 10 units closer to the goal...
				v_check = v_check + (v_direction * 10.0);

				v_down = v_check;
				v_down.z = v_down.z - 1000.0;  // straight down 1000 units

				UTIL_TraceLine(v_check, v_down, ignore_monsters,
					pEntity->v.pContainingEntity, &tr);

				curr_height = tr.flFraction * 1000.0;  // height from ground

				// is the difference in the last height and the current height
				// higher that the jump height?
				if ((last_height - curr_height) > 45.0)
				{
					// can't get there from here...
					return FALSE;
				}

				last_height = curr_height;

				distance = (v_dest - v_check).Length();  // distance from goal
			}

			return TRUE;
		}
	}

	return FALSE;
}


// find the nearest reachable waypoint
int WaypointFindReachable(edict_t *pEntity, float range, int team)
{
	int i, min_index;
	float distance;
	float min_distance;
	TraceResult tr;

	// find the nearest waypoint...

	min_distance = 9999.0;

	for (i=0; i < num_waypoints; i++)
	{
		// DrEvils rocketjump pathing
		if ((waypoints[i].flags & W_FL_TFC_JUMP) && (pEntity->v.playerclass != TFC_CLASS_SOLDIER))
			continue;

		if (waypoints[i].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		if (waypoints[i].flags & W_FL_AIMING)
			continue;  // skip any aiming waypoints

		// skip this waypoint if it's team specific and teams don't match...
		if ((team != -1) && (waypoints[i].flags & W_FL_TEAM_SPECIFIC) &&
			((waypoints[i].flags & W_FL_TEAM) != team))
			continue;

		distance = (waypoints[i].origin - pEntity->v.origin).Length();

		if (distance < min_distance)
		{
			// if waypoint is visible from current position (even behind head)...
			UTIL_TraceLine( pEntity->v.origin + pEntity->v.view_ofs, waypoints[i].origin,
				ignore_monsters, pEntity->v.pContainingEntity, &tr );

			if (tr.flFraction >= 1.0)
			{
				if (WaypointReachable(pEntity->v.origin, waypoints[i].origin, pEntity))
				{
					min_index = i;
					min_distance = distance;
				}
			}
		}
	}

	// if not close enough to a waypoint then just return
	if (min_distance > range)
		return -1;

	return min_index;

}


void WaypointPrintInfo(edict_t *pEntity)
{
	char msg[80];
	int index;
	int flags;

	// find the nearest waypoint...
	index = WaypointFindNearest(pEntity, 50.0, -1);

	if (index == -1)
		return;

	sprintf(msg,"Waypoint %d of %d total\n", index, num_waypoints);
	ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);

	flags = waypoints[index].flags;

	if (flags & W_FL_TEAM_SPECIFIC)
	{
		if (mod_id == FRONTLINE_DLL)
		{
			if ((flags & W_FL_TEAM) == 0)
				strcpy(msg, "Waypoint is for Attackers\n");
			else if ((flags & W_FL_TEAM) == 1)
				strcpy(msg, "Waypoint is for Defenders\n");
		}
		else
		{
			if ((flags & W_FL_TEAM) == 0)
				strcpy(msg, "Waypoint is for TEAM 1\n");
			else if ((flags & W_FL_TEAM) == 1)
				strcpy(msg, "Waypoint is for TEAM 2\n");
			else if ((flags & W_FL_TEAM) == 2)
				strcpy(msg, "Waypoint is for TEAM 3\n");
			else if ((flags & W_FL_TEAM) == 3)
				strcpy(msg, "Waypoint is for TEAM 4\n");
		}

		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
	}

	if (flags & W_FL_LIFT)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "Bot will wait for lift before approaching\n");

	if (flags & W_FL_LADDER)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This waypoint is on a ladder\n");

	if (flags & W_FL_DOOR)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a walk waypoint\n");

	if (flags & W_FL_HEALTH)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "There is health near this waypoint\n");

	if (flags & W_FL_ARMOR)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "There is armor near this waypoint\n");

	if (flags & W_FL_AMMO)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "There is ammo near this waypoint\n");

	if (flags & W_FL_SNIPER)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a sniper waypoint\n");

	if (flags & W_FL_JUMP)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a jump waypoint\n");

	if (flags & W_FL_TFC_DEFEND)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a sentry waypoint\n");

	if (flags & W_FL_TFC_DEFEND180)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a 180 degree sentry rotation waypoint\n");

	if (flags & W_FL_TFC_TELEPORTER_ENTRANCE)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a teleporter entrance waypoint\n");

	if (flags & W_FL_TFC_TELEPORTER_EXIT)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a teleporter exit waypoint\n");

	if (flags & W_FL_TFC_PL_DEFEND)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a HW/Soldier defender point\n");

	if (flags & W_FL_TFC_PIPETRAP)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a Demo-man defender point\n");

	if (flags & W_FL_TFC_JUMP)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is a rocket jump waypoint\n");

	if (flags & W_FL_POINT1)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point1\n");

	if (flags & W_FL_POINT2)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point2\n");

	if (flags & W_FL_POINT3)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point3\n");

	if (flags & W_FL_POINT4)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point4\n");

	if (flags & W_FL_POINT5)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point5\n");

	if (flags & W_FL_POINT6)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point6\n");

	if (flags & W_FL_POINT7)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point7\n");

	if (flags & W_FL_POINT8)
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "This is marked as point8\n");

	if (flags & W_FL_TFC_FLAG)
	{
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "There is a flag near this waypoint\n");
	}

	if (flags & W_FL_TFC_FLAG_GOAL)
	{
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "There is a flag goal near this waypoint\n");
	}	
}


void WaypointThink(edict_t *pEntity)
{
	//crashes??
	if(!pEntity || botcamEnabled) return;

	float distance, min_distance;
	Vector start, end;
	int i, index;

	// Auto waypoint as u run.
	if (g_auto_waypoint && !((pEntity->v.flags & FL_FAKECLIENT)==FL_FAKECLIENT))  // is auto waypoint on?
	{
		// find the distance from the last used waypoint
		distance = (last_waypoint - pEntity->v.origin).Length();

		if (distance > 200)
		{
			min_distance = 9999.0;

			// check that no other reachable waypoints are nearby...
			for (i=0; i < num_waypoints; i++)
			{
				if (waypoints[i].flags & W_FL_DELETED)
					continue;

				if (waypoints[i].flags & W_FL_AIMING)
					continue;

				if (WaypointReachable(pEntity->v.origin, waypoints[i].origin, pEntity))
				{
					distance = (waypoints[i].origin - pEntity->v.origin).Length();

					if (distance < min_distance)
						min_distance = distance;
				}
			}

			// make sure nearest waypoint is far enough away...
			if (min_distance >= 200)
				WaypointAdd(pEntity);  // place a waypoint here
		}
	}

	min_distance = 9999.0;

	if ((g_waypoint_on || g_area_def) && !g_waypoint_cache)
	{
		char cmd[255];
		sprintf(cmd,"changelevel %s\n",STRING(gpGlobals->mapname));
		SERVER_COMMAND(cmd);		
	}
	if (g_waypoint_on && g_waypoint_cache && !((pEntity->v.flags & FL_FAKECLIENT)==FL_FAKECLIENT))  // display the waypoints if turned on...
	{
		if(g_path_oneway)
			WaypointRunOneWay(pEntity);
		else if (g_path_twoway)
			WaypointRunTwoWay(pEntity);

		int j=0;
		for (i=0; i < num_waypoints; i++)
		{
			if ((waypoints[i].flags & W_FL_DELETED) == W_FL_DELETED)
				continue;


			distance = (waypoints[i].origin - pEntity->v.origin).Length();
			if (distance < 200)
			{
				//waypoint holographic models + sprites
				if(pEntity==INDEXENT(1) && distance < 800)
			 {
				 int f = waypoints[i].flags;
				 if((f & W_FL_HEALTH) == W_FL_HEALTH ||
					 (f & W_FL_ARMOR) == W_FL_ARMOR ||
					 (f & W_FL_AMMO) == W_FL_AMMO ||
					 (f & W_FL_SNIPER) == W_FL_SNIPER ||
					 (f & W_FL_TFC_FLAG) == W_FL_TFC_FLAG ||
					 (f & W_FL_TFC_FLAG_GOAL) == W_FL_TFC_FLAG_GOAL ||
					 (f & W_FL_TFC_DEFEND) == W_FL_TFC_DEFEND ||
					 (f & W_FL_TFC_TELEPORTER_ENTRANCE) == W_FL_TFC_TELEPORTER_ENTRANCE ||
					 (f & W_FL_TFC_TELEPORTER_EXIT) == W_FL_TFC_TELEPORTER_EXIT || 
					 (f & W_FL_POINT1) == W_FL_POINT1 ||
					 (f & W_FL_POINT2) == W_FL_POINT2 ||
					 (f & W_FL_POINT3) == W_FL_POINT3 ||
					 (f & W_FL_POINT4) == W_FL_POINT4 ||
					 (f & W_FL_POINT5) == W_FL_POINT5 ||
					 (f & W_FL_POINT6) == W_FL_POINT6 ||
					 (f & W_FL_POINT7) == W_FL_POINT7 ||
					 (f & W_FL_POINT8) == W_FL_POINT8 ||
					 (f & W_FL_LIFT) == W_FL_LIFT ||
					 (f & W_FL_JUMP) == W_FL_JUMP ||
					 (f & W_FL_TFC_DEMO) == W_FL_TFC_DEMO ||
					 (f & W_FL_TFC_JUMP) == W_FL_TFC_JUMP ||
					 (f & W_FL_TFC_PL_DEFEND) == W_FL_TFC_PL_DEFEND ||
					 (f & W_FL_TFC_PIPETRAP) == W_FL_TFC_PIPETRAP)
				 {
					 short model;
					 if((f & W_FL_HEALTH) == W_FL_HEALTH)
						{
							model = PRECACHE_MODEL("models/w_medkit.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(64);
							MESSAGE_END();
						}
						if((f & W_FL_ARMOR) == W_FL_ARMOR)
						{
							model = PRECACHE_MODEL("models/r_armor.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_AMMO) == W_FL_AMMO)
						{
							model = PRECACHE_MODEL("models/backpack.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_SNIPER) == W_FL_SNIPER)
						{
							model = PRECACHE_MODEL("models/p_sniper.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_FLAG) == W_FL_TFC_FLAG)
						{
							model = PRECACHE_MODEL("models/flag.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_FLAG_GOAL) == W_FL_TFC_FLAG_GOAL)
						{
							model = PRECACHE_MODEL("models/bigrat.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(64);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_DEFEND) == W_FL_TFC_DEFEND)
						{
							model = PRECACHE_MODEL("models/sentry1.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_PL_DEFEND) == W_FL_TFC_PL_DEFEND)
						{
							model = PRECACHE_MODEL("models/player/hvyweapon/hvyweapon.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_PIPETRAP) == W_FL_TFC_PIPETRAP)
						{
							model = PRECACHE_MODEL("models/player/demo/demo.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}						
						if((f & W_FL_JUMP) == W_FL_JUMP)
						{
							model = PRECACHE_MODEL("models/w_longjump.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_JUMP) == W_FL_TFC_JUMP)
						{
							model = PRECACHE_MODEL("models/rpgrocket.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if((f & W_FL_TFC_DEMO) == W_FL_TFC_DEMO)
						{
							model = PRECACHE_MODEL("models/detpack.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if(((f & W_FL_TFC_TELEPORTER_ENTRANCE) == W_FL_TFC_TELEPORTER_ENTRANCE) || ((f & W_FL_TFC_TELEPORTER_EXIT) == W_FL_TFC_TELEPORTER_EXIT))
						{
							model = PRECACHE_MODEL("models/teleporter.mdl");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						if(((f & W_FL_LIFT) == W_FL_LIFT))
						{
							model = PRECACHE_MODEL("sprites/arrow1.spr");
							MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
							WRITE_BYTE(17);
							WRITE_COORD(waypoints[i].origin.x);
							WRITE_COORD(waypoints[i].origin.y);
							WRITE_COORD(waypoints[i].origin.z);
							WRITE_SHORT(model);
							WRITE_BYTE(0.1);
							WRITE_BYTE(128);
							MESSAGE_END();
						}
						//dots displaying what points are set for a waypoint
						if(
							((f & W_FL_POINT1) == W_FL_POINT1 ||
							(f & W_FL_POINT2) == W_FL_POINT2 ||
							(f & W_FL_POINT3) == W_FL_POINT3 ||
							(f & W_FL_POINT4) == W_FL_POINT4 ||
							(f & W_FL_POINT5) == W_FL_POINT5 ||
							(f & W_FL_POINT6) == W_FL_POINT6 ||
							(f & W_FL_POINT7) == W_FL_POINT7 ||
							(f & W_FL_POINT8) == W_FL_POINT8))
						{
							if((f & W_FL_POINT1) == W_FL_POINT1)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+40);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else							
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+40);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT2) == W_FL_POINT2)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+30);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+30);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT3) == W_FL_POINT3)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+20);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+20);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT4) == W_FL_POINT4)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+10);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z+10);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT5) == W_FL_POINT5)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT6) == W_FL_POINT6)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-10);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-10);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT7) == W_FL_POINT7)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-20);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-20);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}
							if((f & W_FL_POINT8) == W_FL_POINT8)
							{
								model = PRECACHE_MODEL("sprites/cnt1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-30);
								WRITE_SHORT(model);
								WRITE_BYTE(2);
								WRITE_BYTE(64);
								MESSAGE_END();
							}
							else
							{
								model = PRECACHE_MODEL("sprites/gargeye1.spr");
								MESSAGE_BEGIN(MSG_PAS, SVC_TEMPENTITY, pEntity->v.origin);
								WRITE_BYTE(17);
								WRITE_COORD(waypoints[i].origin.x);
								WRITE_COORD(waypoints[i].origin.y);
								WRITE_COORD(waypoints[i].origin.z-30);
								WRITE_SHORT(model);
								WRITE_BYTE(3);
								WRITE_BYTE(128);
								MESSAGE_END();
							}

						}
				 }
			 }
			 if (distance < min_distance)
			 {
				 index = i; // store index of nearest waypoint
				 min_distance = distance;
			 }

			 if ((wp_display_time[i] + 1.0) < gpGlobals->time)
			 {
				 if (waypoints[i].flags & W_FL_CROUCH)
				 {
					 start = waypoints[i].origin - Vector(0, 0, 17);
					 end = start + Vector(0, 0, 34);
				 }
				 else if (waypoints[i].flags & W_FL_AIMING)
				 {
					 start = waypoints[i].origin + Vector(0, 0, 10);
					 end = start + Vector(0, 0, 14);
				 }
				 else
				 {
					 start = waypoints[i].origin - Vector(0, 0, 34);
					 end = start + Vector(0, 0, 68);
				 }

				 // draw a blue waypoint
				 int r,g,b;
				 r=150;
				 g=150;
				 b=150;
				 if (waypoints[i].flags & W_FL_TEAM_SPECIFIC)
				 {
					 if ((waypoints[i].flags & W_FL_TEAM) == 0)
					 {
						 r=0; g=0; b=255;
					 }
					 if ((waypoints[i].flags & W_FL_TEAM) == 1)
					 {
						 r=255; g=0; b=0;
					 }
					 if ((waypoints[i].flags & W_FL_TEAM) == 2)
					 {
						 r=255; g=255; b=0; //yellow team
					 }
					 if ((waypoints[i].flags & W_FL_TEAM) == 3)
					 {
						 r=0; g=255; b=0;
					 }
				 }
				 WaypointDrawBeam(pEntity, start, end, 30, 0, r, g, b, 250, 5);
				 //ok, last but not least, jump and lift waypoints..

				 if(is_junction[i])
				 {
					 WaypointDrawBeam(pEntity, end+Vector(0,-10,20), end+Vector(0,10,20), 30, 0, 255, 255, 255, 250, 5);
					 WaypointDrawBeam(pEntity, end+Vector(-10,0,20), end+Vector(10,0,20), 30, 0, 255, 255, 255, 250, 5);
				 }

				 wp_display_time[i] = gpGlobals->time;
			 }
			}
		}
		//end of for loop...

		// check if path waypointing is on...
		if (g_path_waypoint)
		{
			// check if player is close enough to a waypoint and time to draw path...
			if ((min_distance <= 50) && (f_path_time <= gpGlobals->time))
			{
				PATH *p;

				f_path_time = gpGlobals->time + 1.0;

				//reverse paths..
				//paths To the one your stood at..
				for (int ii=0; ii < num_waypoints; ii++)
				{
					//only include ones close enoughf
					if((waypoints[index].origin-waypoints[ii].origin).Length()<800)
					{
						p = paths[ii];
						while (p != NULL)
						{
							i = 0;
							while (i < MAX_PATH_INDEX)
							{
								//goes to waypoint were stood at
								if (p->index[i] != -1 && p->index[i]==index)
						  {
							  Vector v_src = waypoints[ii].origin-Vector(0,0,16);
							  Vector v_dest = waypoints[p->index[i]].origin-Vector(0,0,16);
							  WaypointDrawBeam(pEntity, v_src, v_dest, 10, 2, 200, 200, 0, 200, 10);
						  }
						  i++;
							}
							p = p->next;  // go to next node in linked list
						}
					}
				}

				p = paths[index];

				while (p != NULL)
				{
					i = 0;

					while (i < MAX_PATH_INDEX)
					{
						if (p->index[i] != -1)
						{
							Vector v_src = waypoints[index].origin;
							Vector v_dest = waypoints[p->index[i]].origin;

							// draw a white line to this index's waypoint
							WaypointDrawBeam(pEntity, v_src, v_dest, 10, 2, 250, 250, 250, 200, 10);
						}

						i++;
					}

					p = p->next;  // go to next node in linked list
				}
			}
		}
	}
	//mdls=g_waypoint_on;
	//my debug stuff
	if(g_find_waypoint && !((pEntity->v.flags & FL_FAKECLIENT)==FL_FAKECLIENT))
		WaypointDrawBeam(pEntity, pEntity->v.origin,waypoints[g_find_wp].origin, 10, 2, 250, 250, 0, 200, 10);
	//area def draw
	min_distance = 9999.0;
	if (g_area_def && g_waypoint_cache && !((pEntity->v.flags & FL_FAKECLIENT)==FL_FAKECLIENT))
	{
		bool timr;
		for (i=0; i < num_areas; i++)
		{
			//checks only for whole area delete.....
			if ((areas[i].flags & W_FL_DELETED) == W_FL_DELETED)
				continue;
			timr=false;
			if ((areas[i].flags & A_FL_1) == A_FL_1)
		 {
			 //display 1 of 4..(a)
			 distance = (areas[i].a - pEntity->v.origin).Length();

			 if (distance < 300)
			 {
				 if (distance < min_distance)
				 {
					 index = i; // store index of nearest waypoint
					 min_distance = distance;
				 }

				 if ((a_display_time[i] + 1.0) < gpGlobals->time)
				 {
					 start = areas[i].a - Vector(0, 0, 34);
					 end = start + Vector(0, 0, 68);
					 // draw a red waypoint
					 WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

					 timr=true;
				 }
			 }
		 }
		 if ((areas[i].flags & A_FL_2) == A_FL_2)
		 {
			 //display 2 of 4..(b)
			 distance = (areas[i].b - pEntity->v.origin).Length();

			 if (distance < 300)
			 {
				 if (distance < min_distance)
				 {
					 index = i; // store index of nearest waypoint
					 min_distance = distance;
				 }

				 if ((a_display_time[i] + 1.0) < gpGlobals->time)
				 {
					 start = areas[i].b - Vector(0, 0, 34);
					 end = start + Vector(0, 0, 68);
					 // draw a red waypoint
					 WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

					 timr=true;
				 }
			 }
		 }
		 if ((areas[i].flags & A_FL_3) == A_FL_3)
		 {
			 //display 3 of 4..(c)
			 distance = (areas[i].c - pEntity->v.origin).Length();

			 if (distance < 300)
			 {
				 if (distance < min_distance)
				 {
					 index = i; // store index of nearest waypoint
					 min_distance = distance;
				 }

				 if ((a_display_time[i] + 1.0) < gpGlobals->time)
				 {
					 start = areas[i].c - Vector(0, 0, 34);
					 end = start + Vector(0, 0, 68);
					 // draw a red waypoint
					 WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

					 timr=true;
				 }
			 }
		 }
		 if ((areas[i].flags & A_FL_4) == A_FL_4)
		 {
			 //display 4 of 4..(d)
			 distance = (areas[i].d - pEntity->v.origin).Length();

			 if (distance < 300)
			 {
				 if (distance < min_distance)
				 {
					 index = i; // store index of nearest waypoint
					 min_distance = distance;
				 }

				 if ((a_display_time[i] + 1.0) < gpGlobals->time)
				 {
					 start = areas[i].d - Vector(0, 0, 34);
					 end = start + Vector(0, 0, 68);
					 // draw a red waypoint
					 WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

					 timr=true;
				 }
			 }
		 }


		 //connect the dots :)
		 if ((a_display_time[i] + 1.0) < gpGlobals->time && timr)
		 {
			 int g;
			 g=0;
			 //if(AreaInside(pEntity,i)) g=64;
			 if(AreaInsideClosest(pEntity)==i) 
			 {
				 g=255;
				 if(last_area!=i && pEntity==INDEXENT(1))
				 {
					 AreaDefPrintInfo(pEntity);
					 last_area=i;
				 }
			 }
			 if ((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2)
				 WaypointDrawBeam(pEntity, areas[i].a + Vector(0, 0, 34), areas[i].b + Vector(0, 0, 34), 30, 0, 255, g, 0, 250, 5);
			 if ((areas[i].flags & A_FL_2) == A_FL_2 && (areas[i].flags & A_FL_3) == A_FL_3)
				 WaypointDrawBeam(pEntity, areas[i].b + Vector(0, 0, 34), areas[i].c + Vector(0, 0, 34), 30, 0, 255, g, 0, 250, 5);
			 if ((areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4)
				 WaypointDrawBeam(pEntity, areas[i].c + Vector(0, 0, 34), areas[i].d + Vector(0, 0, 34), 30, 0, 255, g, 0, 250, 5);
			 if ((areas[i].flags & A_FL_4) == A_FL_4 && (areas[i].flags & A_FL_1) == A_FL_1)
				 WaypointDrawBeam(pEntity, areas[i].d + Vector(0, 0, 34), areas[i].a + Vector(0, 0, 34), 30, 0, 255, g, 0, 250, 5);
			 //ok, has this area been fully labeled?
			 //display coloured lines to represent what labels have been done
			 if((areas[i].flags & A_FL_1)==A_FL_1 && (areas[i].flags & A_FL_2)==A_FL_2 &&
				 (areas[i].flags & A_FL_3)==A_FL_3 && (areas[i].flags & A_FL_4)==A_FL_4)
			 {
				 //{ fp=fopen("bot.txt","a"); fprintf(fp, "i: %d\n",i); fclose(fp); }
				 //{ fp=fopen("bot.txt","a"); fprintf(fp, "a: %s\n",areas[i].namea); fclose(fp); }
				 //if(areas[i].namea[0]!=NULL)
				 //WaypointDrawBeam(pEntity, areas[i].a + Vector(0, 0, 40), areas[i].a + Vector(0, 0, 50), 30, 0, 0, 0, 255, 250, 5);
				 //{ fp=fopen("bot.txt","a"); fprintf(fp, "b: %s\n",areas[i].nameb); fclose(fp); }
				 //if(areas[i].nameb[0]!=NULL)
				 //WaypointDrawBeam(pEntity, areas[i].b + Vector(0, 0, 40), areas[i].b + Vector(0, 0, 50), 30, 0, 255, 0, 0, 250, 5);
				 //{ fp=fopen("bot.txt","a"); fprintf(fp, "c: %s\n",areas[i].namec); fclose(fp); }
				 //if(areas[i].namec[0]!=NULL)
				 //WaypointDrawBeam(pEntity, areas[i].c + Vector(0, 0, 40), areas[i].c + Vector(0, 0, 50), 30, 0, 255, 255, 0, 250, 5);
				 //{ fp=fopen("bot.txt","a"); fprintf(fp, "d: %s\n",areas[i].named); fclose(fp); }
				 //if(areas[i].named[0]!=NULL)
				 //WaypointDrawBeam(pEntity, areas[i].d + Vector(0, 0, 40), areas[i].d + Vector(0, 0, 50), 30, 0, 0, 255, 0, 250, 5);
			 }
		 }
		 if(timr) a_display_time[i] = gpGlobals->time;

		}
	}
	area_on_last=g_area_def;
}


// run Floyd's algorithm on the waypoint list to generate the least cost
// path matrix...
void WaypointFloyds(unsigned int *shortest_path, unsigned int *from_to)
{
	unsigned int x, y, z, distance;
	unsigned int yRx, yRz, xRz;
	bool changed = true;

	for (y=0; y < route_num_waypoints; y++)
	{
		for (z=0; z < route_num_waypoints; z++)
		{
			from_to[y * route_num_waypoints + z] = z;
		}
	}

	while (changed)
	{
		changed = false;
		for (x=0; x < route_num_waypoints; x++)
		{
			for (y=0; y < route_num_waypoints; y++)
			{
				yRx = y * route_num_waypoints + x;
				for (z = 0; z < route_num_waypoints; z++)
				{
					// Optimization - didn't need to do that math all over the place in the [], just do it once.					
					yRz = y * route_num_waypoints + z;
					xRz = x * route_num_waypoints + z;

					if ((shortest_path[yRx] == WAYPOINT_UNREACHABLE) || (shortest_path[xRz] == WAYPOINT_UNREACHABLE))
						continue;

					distance = shortest_path[yRx] + shortest_path[xRz];

					if (distance > WAYPOINT_MAX_DISTANCE)
						distance = WAYPOINT_MAX_DISTANCE;

					if ((distance < shortest_path[yRz]) || (shortest_path[yRz] == WAYPOINT_UNREACHABLE))
					{
						shortest_path[yRz] = distance;
						from_to[yRz] = from_to[yRx];
						changed = true;
					}
				}
			}
		}
	}
}


// load the waypoint route files (.fw1, .fw2, etc.) or generate them if
// they don't exist...
void WaypointRouteInit(void)
{
	unsigned int index;
	bool build_matrix[4];
	int matrix;
	unsigned int array_size;
	unsigned int row;
	int i, offset;
	unsigned int a, b;
	float distance;
	unsigned int *pShortestPath, *pFromTo;
	char msg[80];
	//unsigned int num_items;
	//FILE *bfp;
	char filename[256];
	char filename2[256];
	char mapname[64];

	if (num_waypoints == 0)
		return;

	// save number of current waypoints in case waypoints get added later
	route_num_waypoints = num_waypoints;

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".fwp");

	UTIL_BuildFileName(filename, "waypoints", mapname);

	build_matrix[0] = TRUE;  // always build matrix 0 (non-team and team 1)
	build_matrix[1] = FALSE;
	build_matrix[2] = FALSE;
	build_matrix[3] = FALSE;

	// find out how many route matrixes to create...
	for (index=0; index < route_num_waypoints; index++)
	{
		if (waypoints[index].flags & W_FL_TEAM_SPECIFIC)
		{
			if ((waypoints[index].flags & W_FL_TEAM) == 0x01)  // team 2?
				build_matrix[1] = TRUE;

			if ((waypoints[index].flags & W_FL_TEAM) == 0x02)  // team 3?
				build_matrix[2] = TRUE;

			if ((waypoints[index].flags & W_FL_TEAM) == 0x03)  // team 4?
				build_matrix[3] = TRUE;
		}
	}

	array_size = route_num_waypoints * route_num_waypoints;

	for (matrix=0; matrix < 4; matrix++)
	{
		if (build_matrix[matrix])
		{
			char ext_str[5];  // ".fwX\0"
			//int file1, file2;
			//struct stat stat1, stat2;

			sprintf(ext_str, ".fw%d", matrix+1);

			strcpy(mapname, STRING(gpGlobals->mapname));
			strcat(mapname, ext_str);

			UTIL_BuildFileName(filename2, "waypoints", mapname);

			if (shortest_path[matrix] == NULL)
			{
				sprintf(msg, "calculating FoXBot waypoint paths for team %d...\n", matrix+1);
				ALERT(at_console, msg);

				shortest_path[matrix] = (unsigned int *)malloc(sizeof(unsigned int) * array_size);

				if (shortest_path[matrix] == NULL)
					ALERT(at_error, "FoXBot - Error allocating memory for shortest path!");

				from_to[matrix] = (unsigned int *)malloc(sizeof(unsigned int) * array_size);

				if (from_to[matrix] == NULL)
					ALERT(at_error, "FoXBot - Error allocating memory for from to matrix!");

				pShortestPath = shortest_path[matrix];
				pFromTo = from_to[matrix];

				for (index=0; index < array_size; index++)
					pShortestPath[index] = WAYPOINT_UNREACHABLE;

				for (index=0; index < route_num_waypoints; index++)
					pShortestPath[index * route_num_waypoints + index] = 0;  // zero diagonal

				for (row=0; row < route_num_waypoints; row++)
				{
					if (paths[row] != NULL)
					{
						PATH *p = paths[row];

						while (p)
						{
							i = 0;

							while (i < MAX_PATH_INDEX)
							{
								if (p->index[i] != -1)
								{
									index = p->index[i];

									// check if this is NOT team specific OR matches this team
									if (!(waypoints[index].flags & W_FL_TEAM_SPECIFIC) ||
										((waypoints[index].flags & W_FL_TEAM) == matrix))
									{
										distance = (waypoints[row].origin - waypoints[index].origin).Length();

										if (distance > (float)WAYPOINT_MAX_DISTANCE)
											distance = (float)WAYPOINT_MAX_DISTANCE;

										if (distance > REACHABLE_RANGE)
										{
											sprintf(msg, "Waypoint path distance > %4.1f at from %d to %d\n",
												REACHABLE_RANGE, row, index);
											ALERT(at_console, msg);
											WaypointDeletePath(row,index);
										}
										else
										{
											offset = row * route_num_waypoints + index;

											pShortestPath[offset] = (unsigned int)distance;
										}
									}
								}

								i++;
							}

							p = p->next;  // go to next node in linked list
						}
					}
				}

				// run Floyd's Algorithm to generate the from_to matrix...
				WaypointFloyds(pShortestPath, pFromTo);

				for (a=0; a < route_num_waypoints; a++)
				{
					for (b=0; b < route_num_waypoints; b++)
						if (pShortestPath[a * route_num_waypoints + b] == WAYPOINT_UNREACHABLE)
							pFromTo[a * route_num_waypoints + b] = WAYPOINT_UNREACHABLE;
				}
				sprintf(msg, "FoXBot waypoint path calculations for team %d complete!\n",matrix+1);
				ALERT(at_console, msg);
			}
		}
	}
	ALERT(at_console, "Loading Rocket/Conc Jump waypoints...");
	int RJIndex = 0;
	// a local array to keep track of the # of RJ points for each team
	int teamCount[4];
	for (i = 0; i < 4; i++)
		teamCount[i] = 0;
	// initialize the RJPoints to -1 so we can check for that later to stop further checking
	for (i = 0; i < bot_t::MAXRJWAYPOINTS; i++)
	{
		RJPoints[i][0] = -1;
		RJPoints[i][1] = -1;
	}
	// Go through the waypoints and look for rj/conc wps
	for (i = 0; i < num_waypoints; i++)
	{
		// Skip deleted waypoints.
		if ((waypoints[i].flags & W_FL_DELETED)==W_FL_DELETED)
			continue;

		// Is it a rocket jump point?
		if ((waypoints[i].flags & W_FL_TFC_JUMP)==W_FL_TFC_JUMP)
		{
			// Check any team specific flags
			if (waypoints[i].flags & W_FL_TEAM_SPECIFIC) 
			{
				// Put the waypoint index, and the team # into the list.
				RJPoints[RJIndex][0] = i;
				RJPoints[RJIndex][1] = (waypoints[index].flags & W_FL_TEAM);  
				teamCount[RJPoints[RJIndex][1]-1]++;
				RJIndex++;
			} else
			{
				// Just store the index, leave the team field at -1, means no team specific
				RJPoints[RJIndex][0] = i;
				RJIndex++;
			}
			// Dont go over the limit
			if (RJIndex == bot_t::MAXRJWAYPOINTS)
				break;
		}	   
	}
	sprintf(msg, "RJ/Conc Total: %d : Blue: %d : Red: %d : Yellow: %d : Green: %d\n", RJIndex+1, teamCount[0], teamCount[1], teamCount[2], teamCount[3]);
	ALERT(at_console, msg);
}


// return the next waypoint index for a path from the Floyd matrix when
// going from a source waypoint index (src) to a destination waypoint
// index (dest)...
unsigned short WaypointRouteFromTo(int src, int dest, int team)
{
	unsigned int *pFromTo;

	if ((team < -1) || (team > 3))
		return -1;

	if (team == -1)  // -1 means non-team play
		team = 0;

	if (from_to[team] == NULL)  // if no team specific waypoints use team 0
		team = 0;

	if (from_to[team] == NULL)  // if no route information just return
		return -1;

	pFromTo = from_to[team];

	return pFromTo[src * route_num_waypoints + dest];
}


// return the total distance (based on the Floyd matrix) of a path from
// the source waypoint index (src) to the destination waypoint index
// (dest)...
int WaypointDistanceFromTo(int src, int dest, int team)
{
	unsigned int *pShortestPath;

	if ((team < -1) || (team > 3))
		return -1;

	if (team == -1)  // -1 means non-team play
		team = 0;

	if (from_to[team] == NULL)  // if no team specific waypoints use team 0
		team = 0;

	if (from_to[team] == NULL)  // if no route information just return
		return -1;

	pShortestPath = shortest_path[team];

	return (int)(pShortestPath[src * route_num_waypoints + dest]);
}

//distances are reversed with this function...
//in that the number returned is greater as you get closer to dest
int WaypointMultiDistanceFromTo(bot_t *pBot,int src, int dest, int team)
{
	unsigned short int *pShortestPath;

	if ((team < -1) || (team > 3))
		return -1;

	if (team == -1)  // -1 means non-team play
		team = 0;

	if (from_to[team] == NULL)  // if no team specific waypoints use team 0
		team = 0;

	if (from_to[team] == NULL)  // if no route information just return
		return -1;

	//pShortestPath = pBot->multi_path_dist;

	//if((pShortestPath[src * route_num_waypoints + dest])==USHRT_MAX)
	return -1;
	return (int)(pShortestPath[src * route_num_waypoints + dest]);
}

//index is wpt, ii=team 
//checks to see if the wpt has points set, is it available or not
bool WaypointAvailable(int index,int ii)
{
	int ij=-1;
	//as we get the nearest team wpt and pass it in
	//if theirs not a team specific wpt their, return false
	//if(index==-1) return false;
	//hmm, will fuck it if theirs no wpt near the item! grr
	if(index!=-1 && index<num_waypoints && ii!=-1)
	{
		bool fc=true;
		bool fcc=false;
		if((waypoints[index].flags & W_FL_POINT1) == W_FL_POINT1) 
		{
			fcc=true;
			ij=0;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT2) == W_FL_POINT2) 
		{
			ij=1;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT3) == W_FL_POINT3) 
		{
			ij=2;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT4) == W_FL_POINT4) 
		{
			ij=3;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT5) == W_FL_POINT5) 
		{
			ij=4;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT6) == W_FL_POINT6) 
		{
			ij=5;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT7) == W_FL_POINT7) 
		{
			ij=6;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if((waypoints[index].flags & W_FL_POINT8) == W_FL_POINT8) 
		{
			ij=7;
			fcc=true;
			switch(ii)
			{
			case 0: if(blue_av[ij]==true) fc=false;
				break;
			case 1: if(red_av[ij]==true) fc=false;
				break;
			case 2: if(yellow_av[ij]==true) fc=false;
				break;
			case 3: if(green_av[ij]==true) fc=false;
				break;
			}
		}
		if(fc && fcc) return false;
	}
	if ((ii != -1) && (waypoints[index].flags & W_FL_TEAM_SPECIFIC) &&
		((waypoints[index].flags & W_FL_TEAM) != ii))
		return false;
	return true;
}

void WaypointRunOneWay(edict_t *pEntity)
{
	int temp = WaypointFindNearest(pEntity, 50.0, -1);

	if (temp != -1)
	{
		if(wpt1==-1)
		{
			// play "cancelled" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_moveselect.wav", 1.0,
				ATTN_NORM, 0, 100);
			wpt1=temp;
			return;
		}
		else
		{
			wpt2=wpt1;
			wpt1=temp;
			if(wpt1!=-1 && wpt2!=-1 && wpt1!=wpt2)
			{
				// play "error" sound...
				EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_select.wav", 1.0,
					ATTN_NORM, 0, 100);

				WaypointAddPath(wpt2,wpt1);
			}
			return;
		}
	}
	return;

}

void WaypointRunTwoWay(edict_t *pEntity)
{
	int temp = WaypointFindNearest(pEntity, 50.0, -1);

	if (temp != -1)
	{
		if(wpt1==-1)
		{
			// play "cancelled" sound...
			EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_moveselect.wav", 1.0,
				ATTN_NORM, 0, 100);
			wpt1=temp;
			return;
		}
		else
		{
			wpt2=wpt1;
			wpt1=temp;
			if(wpt1!=-1 && wpt2!=-1 && wpt1!=wpt2)
			{
				// play "error" sound...
				EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "common/wpn_select.wav", 1.0,
					ATTN_NORM, 0, 100);

				WaypointAddPath(wpt1,wpt2);
				WaypointAddPath(wpt2,wpt1);
			}
			return;
		}
	}
	return;
}

void WaypointAutoBuild(edict_t *pEntity)
{
	int index;
	edict_t *pent = NULL;
	float radius = 40;
//	char item_name[64];
	//clear all wpts
	WaypointInit();

	for(int x=(-4096+32);x<=(4096-32);x=x+32)
	{
		//{ fp=fopen("bot.txt","a"); fprintf(fp,"%d %d\n\n",x,num_waypoints); fclose(fp); }
		for(int y=(-4096+32);y<=(4096-32);y=y+32)
		{
			for(int z=(-4096+(32+32));z<=(4096-32);z=z+32)
			{
				//{ fp=fopen("bot.txt","a"); fprintf(fp,"-%d %d\n",y,num_waypoints); fclose(fp); }
				TraceResult tr;
				//Vector(x,y,(-4096+32))
				//UTIL_TraceLine(Vector(x,y,z),Vector(x,y,z-64)
				//,ignore_monsters,ignore_glass,pEntity,&tr);
				TRACE_HULL(Vector(x,y,z),Vector(x,y,z-32)
					,ignore_monsters,head_hull,pEntity,&tr);
				//{ fp=fopen("bot.txt","a"); fprintf(fp,"%f\n",tr.flFraction); fclose(fp); }
				//while(tr.flFraction<1)
				if(tr.flFraction<1)
				{
					if (num_waypoints >= MAX_WAYPOINTS)
						return;
					//make sure player can atleast duck in this space!
					//TraceResult tr2;
					//UTIL_TraceLine(tr.vecEndPos,tr.vecEndPos+Vector(0,0,32)
					//,ignore_monsters,ignore_glass,pEntity,&tr2);
					//TRACE_HULL(tr.vecEndPos,tr.vecEndPos+Vector(0,0,32)
					//,ignore_monsters,head_hull,pEntity,&tr2);
					//if(tr2.flFraction>=1)
					{
						index = 0;

						// find the next available slot for the new waypoint...
						while (index < num_waypoints)
						{
							if ((waypoints[index].flags & W_FL_DELETED)==W_FL_DELETED)
							{
							 break;
							}

							index++;
						}

						waypoints[index].flags = 0;

						// store the origin (location) of this waypoint (use entity origin)
						waypoints[index].origin = tr.vecEndPos+Vector(0,0,16);

						// store the last used waypoint for the auto waypoint code...
						//last_waypoint = tr.vecEndPos;

						// set the time that this waypoint was originally displayed...
						wp_display_time[index] = gpGlobals->time;


						Vector start, end;

						start = tr.vecEndPos - Vector(0,0,34);
						end = start + Vector(0, 0, 68);


						//********************************************************
						// look for lift, ammo, flag, health, armor, etc.
						//********************************************************

						/*while ((pent = FIND_ENTITY_IN_SPHERE( pent, tr.vecEndPos, radius )) != NULL &&
						(!FNullEnt(pent)))
						{
						strcpy(item_name, STRING(pent->v.classname));

						if (strcmp("item_healthkit", item_name) == 0)
						{
						//ClientPrint(pEntity, HUD_PRINTCONSOLE, "found a healthkit!\n");
						waypoints[index].flags |= W_FL_HEALTH;
						}

						if (strncmp("item_armor", item_name, 10) == 0)
						{
						//ClientPrint(pEntity, HUD_PRINTCONSOLE, "found some armor!\n");
						waypoints[index].flags |= W_FL_ARMOR;
						}

						// *************
						// LOOK FOR AMMO
						// *************

						}*/

						// draw a blue waypoint
						//WaypointDrawBeam(pEntity, start, end, 30, 0, 0, 0, 255, 250, 5);

						// increment total number of waypoints if adding at end of array...
						if (index == num_waypoints)
							num_waypoints++;

						//delete stuff just incase.
						WaypointDeletePath(index);
						//delete from index to other..cus his delete function dont!

						PATH *p;
						int  i;
						p = paths[index];
						int count = 0;
						// search linked list for del_index...
						while (p != NULL)
						{
							i = 0;
							while (i < MAX_PATH_INDEX)
						 {
							 p->index[i] = -1;  // unassign this path
							 i++;
						 }
						 p = p->next;  // go to next node in linked list

#ifdef _DEBUG
						 count++;
						 if (count > 100) WaypointDebug();
#endif
						}

						/*if(g_path_connect)
						{
						// calculate all the paths to this new waypoint
						for (int i=0; i < num_waypoints; i++)
						{
						if (i == index)
						continue;  // skip the waypoint that was just added

						if (waypoints[i].flags & W_FL_AIMING)
						continue;  // skip any aiming waypoints

						if ((waypoints[i].flags & W_FL_DELETED)==W_FL_DELETED)
						continue; //skip deleted wpts!!

						// check if the waypoint is reachable from the new one (one-way)
						if ( WaypointReachable(waypoints[index].origin, waypoints[i].origin, pEntity) )
						{
						WaypointAddPath(index, i);
						}

						// check if the new one is reachable from the waypoint (other way)
						if ( WaypointReachable(waypoints[i].origin, waypoints[index].origin, pEntity) )
						{
						WaypointAddPath(i, index);
						}
						}
						}*/
					}
					//now do the next trace
					/*				  if(tr.vecEndPos.z-32>(-4096+32))
					{
					UTIL_TraceLine(tr.vecEndPos-Vector(0,0,32),Vector(x,y,(-4096+32))
					,ignore_monsters,ignore_glass,pEntity,&tr);
					}
					else
					tr.flFraction=1;*/
				}
			}
		}
	}
}

//area def stuff..

void AreaDefCreate(edict_t *pEntity)
{

	int index;

	if (num_areas >= MAX_WAYPOINTS)
		return;

	index = 0;

	// find the next available slot for the new waypoint...
	while (index < num_areas)
	{
		if ((areas[index].flags & W_FL_DELETED)==W_FL_DELETED)
		{
			break;
		}
		//stop here if we havn't finished filling an area wif points
		if (!((areas[index].flags & A_FL_1) == A_FL_1 && (areas[index].flags & A_FL_2) == A_FL_2 &&
			(areas[index].flags & A_FL_3) == A_FL_3 && (areas[index].flags & A_FL_4) == A_FL_4))
		{
			break;
		}
		index++;
	}

	if(index==num_areas) num_areas++;
	//areas[index].flags = 0;

	if(!((areas[index].flags & A_FL_1) == A_FL_1))
	{
		areas[index].a = pEntity->v.origin;
		areas[index].flags |= A_FL_1;
		// set the time that this waypoint was originally displayed...
		a_display_time[index] = gpGlobals->time;

		Vector start, end;

		start = pEntity->v.origin - Vector(0, 0, 34);
		end = start + Vector(0, 0, 68);

		// draw a blue waypoint
		WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}

	if(!((areas[index].flags & A_FL_2) == A_FL_2))
	{
		areas[index].b = pEntity->v.origin;
		areas[index].flags |= A_FL_2;
		// set the time that this waypoint was originally displayed...
		a_display_time[index] = gpGlobals->time;

		Vector start, end;

		start = pEntity->v.origin - Vector(0, 0, 34);
		end = start + Vector(0, 0, 68);

		// draw a blue waypoint
		WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}

	if(!((areas[index].flags & A_FL_3) == A_FL_3))
	{
		areas[index].c = pEntity->v.origin;
		areas[index].flags |= A_FL_3;
		// set the time that this waypoint was originally displayed...
		a_display_time[index] = gpGlobals->time;

		Vector start, end;

		start = pEntity->v.origin - Vector(0, 0, 34);
		end = start + Vector(0, 0, 68);

		// draw a blue waypoint
		WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;   
	}

	if(!((areas[index].flags & A_FL_4) == A_FL_4))
	{
		areas[index].d = pEntity->v.origin;
		areas[index].flags |= A_FL_4;
		// set the time that this waypoint was originally displayed...
		a_display_time[index] = gpGlobals->time;

		Vector start, end;

		start = pEntity->v.origin - Vector(0, 0, 34);
		end = start + Vector(0, 0, 68);

		// draw a blue waypoint
		WaypointDrawBeam(pEntity, start, end, 30, 0, 255, 0, 0, 250, 5);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/xbow_hit1.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}

}

int AreaDefPointFindNearest(edict_t *pEntity, float range, int flags)
{
	int i, min_index;
	float distance;
	float min_distance;
	TraceResult tr;
	Vector o;

	if (num_areas < 1)
		return -1;

	// find the nearest area point (not whole area)...

	min_index = -1;
	min_distance = 9999.0;

	for (i=0; i < num_areas; i++)
	{
		if (areas[i].flags & W_FL_DELETED)
			continue;  // skip any deleted waypoints

		distance=9999;

		if(flags==A_FL_1) 
		{
			distance = (areas[i].a - pEntity->v.origin).Length();
			o=areas[i].a;
		}
		if(flags==A_FL_2) 
		{
			distance = (areas[i].b - pEntity->v.origin).Length();
			o=areas[i].b;
		}
		if(flags==A_FL_3)
		{
			distance = (areas[i].c - pEntity->v.origin).Length();
			o=areas[i].c;
		}
		if(flags==A_FL_4)
		{
			distance = (areas[i].d - pEntity->v.origin).Length();
			o=areas[i].d;
		}

		if(distance==9999) continue;

		if ((distance < min_distance) && (distance < range))
		{
			// if waypoint is visible from current position (even behind head)...
			UTIL_TraceLine( pEntity->v.origin + pEntity->v.view_ofs, o,
				ignore_monsters, pEntity->v.pContainingEntity, &tr );

			if (tr.flFraction >= 1.0)
			{
				min_index = i;
				min_distance = distance;
			}
		}
	}

	return min_index;
}


void AreaDefDelete(edict_t *pEntity)
{
	int index;
	int count = 0;

	if (num_areas < 1)
		return;

	index = AreaDefPointFindNearest(pEntity, 50.0, A_FL_1);
	if (index != -1)
	{
		areas[index].flags &= ~A_FL_1;
		areas[index].a = Vector(0,0,0);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/mine_activate.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}
	index = AreaDefPointFindNearest(pEntity, 50.0, A_FL_2);
	if (index != -1)
	{
		areas[index].flags &= ~A_FL_2;
		areas[index].b = Vector(0,0,0);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/mine_activate.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}
	index = AreaDefPointFindNearest(pEntity, 50.0, A_FL_3);
	if (index != -1)
	{
		areas[index].flags &= ~A_FL_3;
		areas[index].c = Vector(0,0,0);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/mine_activate.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}
	index = AreaDefPointFindNearest(pEntity, 50.0, A_FL_4);
	if (index != -1)
	{
		areas[index].flags &= ~A_FL_4;
		areas[index].d = Vector(0,0,0);

		EMIT_SOUND_DYN2(pEntity, CHAN_WEAPON, "weapons/mine_activate.wav", 1.0,
			ATTN_NORM, 0, 100);
		return;
	}
}

void AreaDefSave(void)
{
	char filename[256];
	char mapname[64];
	AREA_HDR header;
	int index;

	strcpy(header.filetype, "FoXBot");

	header.area_file_version = AREA_VERSION;

	header.number_of_areas = num_areas;

	memset(header.mapname, 0, sizeof(header.mapname));
	strncpy(header.mapname, STRING(gpGlobals->mapname), 31);
	header.mapname[31] = 0;

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".far");

	UTIL_BuildFileName(filename, "areas", mapname);

	FILE *bfp = fopen(filename, "wb");

	// write the waypoint header to the file...
	fwrite(&header, sizeof(header), 1, bfp);

	// write the waypoint data to the file...
	for (index=0; index < num_areas; index++)
	{
		fwrite(&areas[index], sizeof(areas[0]), 1, bfp);
	}

	fclose(bfp);
}

bool AreaDefLoad(edict_t *pEntity)
{
	char mapname[64];
	char filename[256];
	AREA_HDR header;
	char msg[80];
	int i;
	//return false; //test to see if having areas loaded is
	//fucking up the server (mem crashes)

	strcpy(mapname, STRING(gpGlobals->mapname));
	strcat(mapname, ".far");

	FILE *bfp;

	UTIL_BuildFileName(filename, "areas", mapname);
	bfp = fopen(filename, "rb");

	if(bfp==NULL)
	{
		UTIL_BuildFileNameOld(filename, "maps", mapname);
		bfp = fopen(filename, "rb");
	}

	if (bfp != NULL)
	{
		if (IS_DEDICATED_SERVER())
			printf("loading area file: %s\n", filename);
		fread(&header, sizeof(header), 1, bfp);

		header.filetype[7] = 0;
		if (strcmp(header.filetype, "FoXBot") == 0)
		{
			if (header.area_file_version != AREA_VERSION)
			{
				if (pEntity)
					ClientPrint(pEntity, HUD_PRINTNOTIFY, "Incompatible FoXBot area file version!\nAreas not loaded!\n");

				fclose(bfp);
				return FALSE;
			}

			header.mapname[31] = 0;

			if (stricmp(header.mapname, STRING(gpGlobals->mapname)) == 0)
			{
				//works for areas aswell :)
				//WaypointInit();  // remove any existing waypoints
				//grr, removes loaded waypoints! doh
				num_areas = 0;

				for (i=0; i < MAX_WAYPOINTS; i++)
				{
					a_display_time[i] = 0.0;
					areas[i].flags = 0;
					areas[i].a = Vector(0,0,0);
					areas[i].b = Vector(0,0,0);
					areas[i].c = Vector(0,0,0);
					areas[i].d = Vector(0,0,0);
					areas[i].namea[0] = NULL;
					areas[i].nameb[0] = NULL;
					areas[i].namec[0] = NULL;
					areas[i].named[0] = NULL;
				}

				if (pEntity)
					ClientPrint(pEntity, HUD_PRINTNOTIFY, "Loading FoXBot area file\n");
				for (i=0; i < header.number_of_areas; i++)
				{
					fread(&areas[i], sizeof(areas[0]), 1, bfp);
					num_areas++;
				}
			}
			else
			{
				if (pEntity)
				{
					sprintf(msg, "%s FoXBot areas are not for this map!\n", filename);
					ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
				}

				fclose(bfp);
				return FALSE;
			}
		}
		else
		{
			if (pEntity)
			{
				sprintf(msg, "%s is not a FoXBot area file!\n", filename);
				ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
			}

			fclose(bfp);
			return FALSE;
		}

		fclose(bfp);
	}

	return TRUE;
}

void AreaDefPrintInfo(edict_t *pEntity)
{
	char msg[1024];
	int index,i;
	//int flags;


	i=AreaInsideClosest(pEntity);
	if(i!=-1)
	{
		_snprintf(msg,1020,"Area %d of %d total\n", i, num_areas);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		_snprintf(msg,1020,"Name1 = %s\n",areas[i].namea);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		_snprintf(msg,1020,"Name2 = %s\n",areas[i].nameb);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		_snprintf(msg,1020,"Name3 = %s\n",areas[i].namec);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		_snprintf(msg,1020,"Name4 = %s\n",areas[i].named);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
	}

	index = AreaDefPointFindNearest(pEntity, 50.0,A_FL_1);

	if (index != -1)
	{
		sprintf(msg,"Area %d of %d total\n", index, num_areas);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "Area corner 1\n");
		return;
	}

	index = AreaDefPointFindNearest(pEntity, 50.0,A_FL_2);

	if (index != -1)
	{
		sprintf(msg,"Area %d of %d total\n", index, num_areas);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "Area corner 2\n");
		return;
	}

	index = AreaDefPointFindNearest(pEntity, 50.0,A_FL_3);

	if (index != -1)
	{
		sprintf(msg,"Area %d of %d total\n", index, num_areas);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "Area corner 3\n");
		return;
	}

	index = AreaDefPointFindNearest(pEntity, 50.0,A_FL_4);

	if (index != -1)
	{
		sprintf(msg,"Area %d of %d total\n", index, num_areas);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, msg);
		ClientPrint(pEntity, HUD_PRINTNOTIFY, "Area corner 4\n");
		return;
	}
}


bool AreaInside(edict_t *pEntity,int i)
{
	bool inside=false;
	if((areas[i].flags & A_FL_1)==A_FL_1 && (areas[i].flags & A_FL_2)==A_FL_2 &&
		(areas[i].flags & A_FL_3)==A_FL_3 && (areas[i].flags & A_FL_4)==A_FL_4)
	{
		//find highest/lowest (is x,y in bounds of polygon?)
		float x,y,lx,ly,hx,hy;
		x=pEntity->v.origin.x;
		y=pEntity->v.origin.y;
		lx=areas[i].a.x;
		if(areas[i].b.x<lx) lx=areas[i].b.x;
		if(areas[i].c.x<lx) lx=areas[i].c.x;
		if(areas[i].d.x<lx) lx=areas[i].d.x;
		ly=areas[i].a.y;
		if(areas[i].b.y<ly) ly=areas[i].b.y;
		if(areas[i].c.y<ly) ly=areas[i].c.y;
		if(areas[i].d.y<ly) ly=areas[i].d.y;
		hx=areas[i].a.x;
		if(areas[i].b.x>hx) hx=areas[i].b.x;
		if(areas[i].c.x>hx) hx=areas[i].c.x;
		if(areas[i].d.x>hx) hx=areas[i].d.x;
		hy=areas[i].a.y;
		if(areas[i].b.y>hy) hy=areas[i].b.y;
		if(areas[i].c.y>hy) hy=areas[i].c.y;
		if(areas[i].d.y>hy) hy=areas[i].d.y;

		//make sure theirs no rounding errors (I think vect = float..)
		lx=lx-1;
		hx=hx+1;
		ly=ly-1;
		hy=hy+1;
		//now the in-bounds check..
		if(x>=lx && x<=hx && y>=ly && y<=hy)
		{
			if (areas[i].a.y<y && areas[i].b.y>=y
				||  areas[i].b.y<y && areas[i].a.y>=y)
			{
				if (areas[i].a.x+(y-areas[i].a.y)/(areas[i].b.y-areas[i].a.y)*(areas[i].b.x-areas[i].a.x)<x)
					inside=!inside;
			}
			if (areas[i].b.y<y && areas[i].c.y>=y
				||  areas[i].c.y<y && areas[i].b.y>=y)
			{
				if (areas[i].b.x+(y-areas[i].b.y)/(areas[i].c.y-areas[i].b.y)*(areas[i].c.x-areas[i].b.x)<x)
					inside=!inside;
			}
			if (areas[i].c.y<y && areas[i].d.y>=y
				||  areas[i].d.y<y && areas[i].c.y>=y)
			{
				if (areas[i].c.x+(y-areas[i].c.y)/(areas[i].d.y-areas[i].c.y)*(areas[i].d.x-areas[i].c.x)<x)
					inside=!inside;
			}
			if (areas[i].d.y<y && areas[i].a.y>=y
				||  areas[i].a.y<y && areas[i].d.y>=y)
			{
				if (areas[i].d.x+(y-areas[i].d.y)/(areas[i].a.y-areas[i].d.y)*(areas[i].a.x-areas[i].d.x)<x)
					inside=!inside;
			}
		}
	}
	return inside;
}

int AreaInsideClosest(edict_t *pEntity)
{
	int index;
	float distance;
	distance = 9999.0;
	index = -1;
	for (int i=0; i < num_areas; i++)
	{
		if(AreaInside(pEntity,i))
		{
			float lz,hz,a;
			lz=areas[i].a.z;
			if(areas[i].b.z<lz) lz=areas[i].b.z;
			if(areas[i].c.z<lz) lz=areas[i].c.z;
			if(areas[i].d.z<lz) lz=areas[i].d.z;
			hz=areas[i].a.z;
			if(areas[i].b.z>hz) hz=areas[i].b.z;
			if(areas[i].c.z>hz) hz=areas[i].c.z;
			if(areas[i].d.z>hz) hz=areas[i].d.z;		   
			//we want the mid point between hz and lz.. that will be our distance..
			//nearly forgot, the distance revolves around the player!
			a=abs((((hz-lz)/2)+lz)-pEntity->v.origin.z);
			if(a<distance)
			{
				distance=a;
				index=i;
			}
		}
	}
	return index;
}


void AreaAutoBuild1()
{
	int h,i,j,k,index;
	int lc,rc,r,l;
	bool ru,lu,rd,ld;
	int lr,ll;
	for(i=0;i<=num_waypoints;i++)
	{
		if(!((waypoints[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (num_areas >= MAX_WAYPOINTS)
				return;

			index = 0;

			// find the next available slot for the new waypoint...
			while (index < num_areas)
			{
				if ((areas[index].flags & W_FL_DELETED)==W_FL_DELETED)
				{
					break;
				}
				//stop here if we havn't finished filling an area wif points
				if (!((areas[index].flags & A_FL_1) == A_FL_1 && (areas[index].flags & A_FL_2) == A_FL_2 &&
					(areas[index].flags & A_FL_3) == A_FL_3 && (areas[index].flags & A_FL_4) == A_FL_4))
				{
					break;
				}
				index++;
			}

			if(index==num_areas) num_areas++;
			areas[index].flags = A_FL_1+A_FL_2+A_FL_3+A_FL_4;
			areas[index].a=waypoints[i].origin+Vector(-16,-16,0);
			areas[index].b=waypoints[i].origin+Vector(-16,16,0);
			areas[index].c=waypoints[i].origin+Vector(16,16,0);
			areas[index].d=waypoints[i].origin+Vector(16,-16,0);

			waypoints[i].flags=W_FL_DELETED;
			//initial area done
			//now expand it
			//look for its neighbours
			bool expanded=true;

			lc=0;
			rc=0;

			r=0;
			l=0;
			k=0;
			ru=false;
			lu=false;
			rd=false;
			ld=false;
			ll=0;
			lr=0;
			while(k<=num_waypoints)
			{
				if(waypoints[i].origin.y==waypoints[k].origin.y &&
					waypoints[i].origin.z==waypoints[k].origin.z &&
					i!=k)
				{
					if(waypoints[i].origin.x-(32*(lc+1))==waypoints[k].origin.x)
					{
						k=-1;
						lc++;
					}
					if(waypoints[i].origin.x+(32*(rc+1))==waypoints[k].origin.x)
					{
						k=-1;
						rc++;
					}
				}
				else if(waypoints[i].origin.y==waypoints[k].origin.y &&
					waypoints[i].origin.z-16<=waypoints[k].origin.z &&
					waypoints[i].origin.z+16>=waypoints[k].origin.z &&
					!lu && !ld &&
					i!=k)
				{
					if(waypoints[i].origin.x-(32*(lc+1))==waypoints[k].origin.x)
					{
						ll=k;
						k=-1;
						lc++;
						if(waypoints[i].origin.z<=waypoints[k].origin.z)
							ld=true;
						else
							lu=true;
					}
				}
				else if(waypoints[i].origin.y==waypoints[k].origin.y &&
					waypoints[i].origin.z-16<=waypoints[k].origin.z &&
					waypoints[i].origin.z+16>=waypoints[k].origin.z &&
					!ru && !rd && 
					i!=k)
				{
					if(waypoints[i].origin.x+(32*(rc+1))==waypoints[k].origin.x)
					{
						lr=k;
						k=-1;
						rc++;
						if(waypoints[i].origin.z<=waypoints[k].origin.z)
							rd=true;
						else
							ru=true;
					}
				}
				else if(waypoints[i].origin.y==waypoints[k].origin.y &&
					(waypoints[ll].origin.z-16<=waypoints[k].origin.z &&
					waypoints[ll].origin.z+16>=waypoints[k].origin.z) ||
					(waypoints[lr].origin.z-16<=waypoints[k].origin.z &&
					waypoints[lr].origin.z+16>=waypoints[k].origin.z) &&
					(ru || rd) && (ld || lu) &&
					i!=k)
				{
					if(waypoints[i].origin.x-(32*(lc+1))==waypoints[k].origin.x)
					{
						if(waypoints[i].origin.z<=waypoints[k].origin.z && ld)
						{
							ll=k;
							k=-1;
							lc++;
						}
						else if(lu)
						{
							ll=k;
							k=-1;
							lc++;
						}						
					}
					if(waypoints[i].origin.x+(32*(rc+1))==waypoints[k].origin.x)
					{
						if(waypoints[i].origin.z<=waypoints[k].origin.z && rd)
						{
							lr=k;
							k=-1;
							rc++;
						}
						else if(ru)
						{
							lr=k;
							k=-1;
							rc++;
						}
					}
				}
				k++;
			}
			while(expanded)
			{
				expanded=false;
				for(j=0;j<=num_waypoints;j++)
				{
					if(!((waypoints[j].flags & W_FL_DELETED)==W_FL_DELETED))
					{
						//expand via y
						//and no slopeing in z
						if(waypoints[i].origin.x==waypoints[j].origin.x &&
							waypoints[i].origin.z==waypoints[j].origin.z &&
							i!=j)
						{
							//expand one way..
							if(waypoints[j].origin.y-16==areas[index].b.y)
							{
								r=0;
								l=0;
								k=0;
								ru=false;
								lu=false;
								rd=false;
								ld=false;
								ll=0;
								lr=0;
								while(k<=num_waypoints)
								{
									if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z==waypoints[k].origin.z &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											k=-1;
											l++;
										}
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											k=-1;
											r++;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z-16<=waypoints[k].origin.z &&
										waypoints[j].origin.z+16>=waypoints[k].origin.z &&
										!lu && !ld &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											ll=k;
											k=-1;
											l++;
											if(waypoints[j].origin.z<=waypoints[k].origin.z)
												ld=true;
											else
												lu=true;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z-16<=waypoints[k].origin.z &&
										waypoints[j].origin.z+16>=waypoints[k].origin.z &&
										!ru && !rd &&
										j!=k)
									{
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											lr=k;
											k=-1;
											r++;
											if(waypoints[j].origin.z<=waypoints[k].origin.z)
												rd=true;
											else
												ru=true;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										(waypoints[ll].origin.z-16<=waypoints[k].origin.z &&
										waypoints[ll].origin.z+16>=waypoints[k].origin.z) ||
										(waypoints[lr].origin.z-16<=waypoints[k].origin.z &&
										waypoints[lr].origin.z+16>=waypoints[k].origin.z) &&
										(ru || rd) && (ld || lu) &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											if(waypoints[j].origin.z<=waypoints[k].origin.z && ld)
											{
												ll=k;
												k=-1;
												l++;
											}
											else if(lu)
											{
												ll=k;
												k=-1;
												l++;
											}						
										}
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											if(waypoints[j].origin.z<=waypoints[k].origin.z && rd)
											{
												lr=k;
												k=-1;
												r++;
											}
											else if(ru)
											{
												lr=k;
												k=-1;
												r++;
											}
										}
									}
									k++;
								}
								if(rc==r && lc==l)
								{
									areas[index].b.y=waypoints[j].origin.y+16;
									areas[index].c.y=waypoints[j].origin.y+16;
									waypoints[j].flags=W_FL_DELETED;
									expanded=true;
								}
							}
							//expand the other way..
							else if(waypoints[j].origin.y+16==areas[index].a.y)
							{
								r=0;
								l=0;
								k=0;
								ru=false;
								lu=false;
								rd=false;
								ld=false;
								ll=0;
								lr=0;
								while(k<=num_waypoints)
								{
									if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z==waypoints[k].origin.z &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											k=-1;
											l++;
										}
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											k=-1;
											r++;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z-16<=waypoints[k].origin.z &&
										waypoints[j].origin.z+16>=waypoints[k].origin.z &&
										!lu && !ld &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											ll=k;
											k=-1;
											l++;
											if(waypoints[j].origin.z<=waypoints[k].origin.z)
												ld=true;
											else
												lu=true;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										waypoints[j].origin.z-16<=waypoints[k].origin.z &&
										waypoints[j].origin.z+16>=waypoints[k].origin.z &&
										!ru && !rd &&
										j!=k)
									{
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											lr=k;
											k=-1;
											r++;
											if(waypoints[j].origin.z<=waypoints[k].origin.z)
												rd=true;
											else
												ru=true;
										}
									}
									else if(waypoints[j].origin.y==waypoints[k].origin.y &&
										(waypoints[ll].origin.z-16<=waypoints[k].origin.z &&
										waypoints[ll].origin.z+16>=waypoints[k].origin.z) ||
										(waypoints[lr].origin.z-16<=waypoints[k].origin.z &&
										waypoints[lr].origin.z+16>=waypoints[k].origin.z) &&
										(ru || rd) && (ld || lu) &&
										j!=k)
									{
										if(waypoints[j].origin.x-(32*(l+1))==waypoints[k].origin.x)
										{
											if(waypoints[j].origin.z<=waypoints[k].origin.z && ld)
											{
												ll=k;
												k=-1;
												l++;
											}
											else if(lu)
											{
												ll=k;
												k=-1;
												l++;
											}						
										}
										if(waypoints[j].origin.x+(32*(r+1))==waypoints[k].origin.x)
										{
											if(waypoints[j].origin.z<=waypoints[k].origin.z && rd)
											{
												lr=k;
												k=-1;
												r++;
											}
											else if(ru)
											{
												lr=k;
												k=-1;
												r++;
											}
										}
									}
									k++;
								}
								if(rc==r && lc==l)
								{
									areas[index].a.y=waypoints[j].origin.y-16;
									areas[index].d.y=waypoints[j].origin.y-16;
									waypoints[j].flags=W_FL_DELETED;
									expanded=true;
								}
							}
						}
					}
				}
			}
		}
	}
	//now all areas have been created (from all the wpts)
	//their will be lots of parallel areas that can be merged...
	for(i=0;i<=num_areas;i++)
	{
		if (!((areas[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2 &&
				(areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4))
			{
				lc=0;
				rc=0;

				r=0;
				l=0;
				k=0;
				h=0;
				ru=false;
				lu=false;
				rd=false;
				ld=false;
				ll=0;
				lr=0;
				while(k<=num_waypoints)
				{
					if(waypoints[k].origin==areas[i].a+Vector(16,16,0))
					{
						h=k;
						k=num_waypoints;
					}
					k++;
				}
				k=0;

				while(k<=num_waypoints)
				{
					if(waypoints[h].origin.x==waypoints[k].origin.x &&
						waypoints[h].origin.z==waypoints[k].origin.z &&
						h!=k)
					{
						if(waypoints[h].origin.y-(32*(lc+1))==waypoints[k].origin.y)
						{
							k=-1;
							lc++;
						}
						if(waypoints[h].origin.y+(32*(rc+1))==waypoints[k].origin.y)
						{
							k=-1;
							rc++;
						}
					}
					else if(waypoints[h].origin.x==waypoints[k].origin.x &&
						waypoints[h].origin.z-16<=waypoints[k].origin.z &&
						waypoints[h].origin.z+16>=waypoints[k].origin.z &&
						!lu && !ld &&
						i!=k)
					{
						if(waypoints[h].origin.y-(32*(lc+1))==waypoints[k].origin.y)
						{
							ll=k;
							k=-1;
							lc++;
							if(waypoints[h].origin.z<=waypoints[k].origin.z)
								ld=true;
							else
								lu=true;
						}
					}
					else if(waypoints[h].origin.x==waypoints[k].origin.x &&
						waypoints[h].origin.z-16<=waypoints[k].origin.z &&
						waypoints[h].origin.z+16>=waypoints[k].origin.z &&
						!ru && !rd && 
						i!=k)
					{
						if(waypoints[h].origin.y+(32*(rc+1))==waypoints[k].origin.y)
						{
							lr=k;
							k=-1;
							rc++;
							if(waypoints[h].origin.z<=waypoints[k].origin.z)
								rd=true;
							else
								ru=true;
						}
					}
					else if(waypoints[h].origin.x==waypoints[k].origin.x &&
						(waypoints[ll].origin.z-16<=waypoints[k].origin.z &&
						waypoints[ll].origin.z+16>=waypoints[k].origin.z) ||
						(waypoints[lr].origin.z-16<=waypoints[k].origin.z &&
						waypoints[lr].origin.z+16>=waypoints[k].origin.z) &&
						(ru || rd) && (ld || lu) &&
						i!=k)
					{
						if(waypoints[h].origin.y-(32*(lc+1))==waypoints[k].origin.y)
						{
							if(waypoints[h].origin.z<=waypoints[k].origin.z && ld)
							{
								ll=k;
								k=-1;
								lc++;
							}
							else if(lu)
							{
								ll=k;
								k=-1;
								lc++;
							}						
						}
						if(waypoints[h].origin.y+(32*(rc+1))==waypoints[k].origin.y)
						{
							if(waypoints[h].origin.z<=waypoints[k].origin.z && rd)
							{
								lr=k;
								k=-1;
								rc++;
							}
							else if(ru)
							{
								lr=k;
								k=-1;
								rc++;
							}
						}
					}
					k++;
				}
				//for each area
				//scan ever other one, and see if it can be merged with it :D

				//as we expanded each area in the y direction earlier
				//we must now merge them in the x direction
				bool expanded=true;
				while(expanded)
				{
					expanded=false;
					for(j=0;j<=num_areas;j++)
					{
						if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED))
						{
							if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
								(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
							{
								if(i!=j)
								{
									if(areas[i].a==areas[j].d &&
										areas[i].b==areas[j].c)
									{
										r=0;
										l=0;
										k=0;
										h=0;
										while(k<=num_waypoints)
										{
											if(waypoints[k].origin==areas[j].d+Vector(-16,16,0))
											{
												h=k;
												k=num_waypoints;
											}
											k++;
										}
										k=0;
										while(k<=num_waypoints)
										{
											if(waypoints[h].origin.x==waypoints[k].origin.x &&
												waypoints[h].origin.z==waypoints[k].origin.z &&
												h!=k)
											{
												if(waypoints[h].origin.y-(32*(l+1))==waypoints[k].origin.y)
												{
													k=-1;
													l++;
												}
												if(waypoints[h].origin.y+(32*(r+1))==waypoints[k].origin.y)
												{
													k=-1;
													r++;
												}
											}
											else if(waypoints[h].origin.x==waypoints[k].origin.x &&
												waypoints[h].origin.z-16<=waypoints[k].origin.z &&
												waypoints[h].origin.z+16>=waypoints[k].origin.z &&
												!lu && !ld &&
												i!=k)
											{
												if(waypoints[h].origin.y-(32*(l+1))==waypoints[k].origin.y)
												{
													ll=k;
													k=-1;
													l++;
													if(waypoints[h].origin.z<=waypoints[k].origin.z)
														ld=true;
													else
														lu=true;
												}
											}
											else if(waypoints[h].origin.x==waypoints[k].origin.x &&
												waypoints[h].origin.z-16<=waypoints[k].origin.z &&
												waypoints[h].origin.z+16>=waypoints[k].origin.z &&
												!ru && !rd && 
												i!=k)
											{
												if(waypoints[h].origin.y+(32*(r+1))==waypoints[k].origin.y)
												{
													lr=k;
													k=-1;
													r++;
													if(waypoints[h].origin.z<=waypoints[k].origin.z)
														rd=true;
													else
														ru=true;
												}
											}
											else if(waypoints[h].origin.x==waypoints[k].origin.x &&
												(waypoints[ll].origin.z-16<=waypoints[k].origin.z &&
												waypoints[ll].origin.z+16>=waypoints[k].origin.z) ||
												(waypoints[lr].origin.z-16<=waypoints[k].origin.z &&
												waypoints[lr].origin.z+16>=waypoints[k].origin.z) &&
												(ru || rd) && (ld || lu) &&
												i!=k)
											{
												if(waypoints[h].origin.y-(32*(l+1))==waypoints[k].origin.y)
												{
													if(waypoints[h].origin.z<=waypoints[k].origin.z && ld)
													{
														ll=k;
														k=-1;
														l++;
													}
													else if(lu)
													{
														ll=k;
														k=-1;
														l++;
													}						
												}
												if(waypoints[h].origin.y+(32*(r+1))==waypoints[k].origin.y)
												{
													if(waypoints[h].origin.z<=waypoints[k].origin.z && rd)
													{
														lr=k;
														k=-1;
														r++;
													}
													else if(ru)
													{
														lr=k;
														k=-1;
														r++;
													}
												}
											}
											k++;
										}
										if(rc==r && lc==l)
										{
											areas[i].a=areas[j].a;
											areas[i].b=areas[j].b;
											areas[j].flags=W_FL_DELETED;
											expanded=true;
										}
									}
								}
							}
						}
					}
				}
			}
		}
	}
}

void AreaAutoMerge()
{
	int i,j,k;
	//first order the areas into largest areas first [NOT DONE]
	//second, take those areas in order of size (largest first) and [NOT DONE]
	//a) find an area adjacent that expands the area in its smallest direction or
	//b) is just adjacent (and can't find/do a)
	//only group if you can find 2 more areas that make the first 2 into a big square

	//work purely with areas...
	//could use wpts aswell, but loading a saved area grid won't include them
	bool a,b,c,d;
	const int stk_sz=512;
	int stk[stk_sz];
	int stk_cnt;

	for(i=0;i<=num_areas;i++)
	{
		if (!((areas[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2 &&
				(areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4))
			{
				a=false;
				b=false;
				c=false;
				d=false;
				for(j=0;j<=num_areas;j++)
				{
					if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
					{
						if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
							(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
						{	
							if(areas[j].a==areas[i].a ||
								areas[j].b==areas[i].a ||
								areas[j].c==areas[i].a ||
								areas[j].d==areas[i].a)
								a=true;
							if(areas[j].a==areas[i].b ||
								areas[j].b==areas[i].b ||
								areas[j].c==areas[i].b ||
								areas[j].d==areas[i].b)
								b=true;
							if(areas[j].a==areas[i].c ||
								areas[j].b==areas[i].c ||
								areas[j].c==areas[i].c ||
								areas[j].d==areas[i].c)
								c=true;
							if(areas[j].a==areas[i].d ||
								areas[j].b==areas[i].d ||
								areas[j].c==areas[i].d ||
								areas[j].d==areas[i].d)
								d=true;

						}
					}
				}

				//corner areas will have 1 bool (out of a-c) false only
				if((!a && b && c && d) ||
					(a && !b && c && d) ||
					(a && b && !c && d) ||
					(a && b && c && !d))
				{
					//Vector aa,bb;
					//now find all the areas we can merge this one with
					bool merged;
					merged=true;
					while(merged)
					{
						stk_cnt=0;
						merged=false;
						if((areas[i].d.x-areas[i].a.x)>
							(areas[i].b.y-areas[i].a.y))
						{
							//x>y so expand in the y direction
							for(j=0;j<=num_areas;j++)
							{
								if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
								{
									if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
										(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
									{	
										if(!a || !d)
										{
											if(areas[j].a==areas[i].b &&
												areas[j].d==areas[i].c)
											{
												areas[i].b=areas[j].b;
												areas[i].c=areas[j].c;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].a==areas[i].b &&
													(areas[j].d.x<areas[i].c.x))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].d.x<areas[i].c.x) &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].d==areas[i].c &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														areas[i].b=areas[stk[0]].b;
														areas[i].c=areas[j].c;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
										if(!b || !c)
										{
											if(areas[j].b==areas[i].a &&
												areas[j].c==areas[i].d)
											{
												areas[i].a=areas[j].a;
												areas[i].d=areas[j].d;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].b==areas[i].a &&
													(areas[j].c.x<areas[i].d.x))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].c.x<areas[i].d.x) &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].c==areas[i].d &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														areas[i].a=areas[stk[0]].a;
														areas[i].d=areas[j].d;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
									}
								}
							}
						}
						else
						{
							//x<=y so expand in the x direction
							for(j=0;j<=num_areas;j++)
							{
								if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
								{
									if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
										(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
									{	
										if(!a || !b)
										{
											if(areas[j].a==areas[i].d &&
												areas[j].b==areas[i].c)
											{
												areas[i].d=areas[j].d;
												areas[i].c=areas[j].c;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].a==areas[i].d &&
													(areas[j].b.y<areas[i].c.y))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].b.y<areas[i].c.y) &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].b==areas[i].c &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														areas[i].d=areas[stk[0]].d;
														areas[i].c=areas[j].c;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
										if(!d || !c)
										{
											if(areas[j].d==areas[i].a &&
												areas[j].c==areas[i].b)
											{
												areas[i].a=areas[j].a;
												areas[i].b=areas[j].b;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].d==areas[i].a &&
													(areas[j].c.y<areas[i].b.y))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].c.y<areas[i].b.y) &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].c==areas[i].b &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														areas[i].a=areas[stk[0]].a;
														areas[i].b=areas[j].b;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
									}
								}
							}
						}
					}
					//end merged
				}

			}
		}
	}

	//clear the remaning shit up!!
	for(i=0;i<=num_areas;i++)
	{
		if (!((areas[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2 &&
				(areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4))
			{
				a=false;
				b=false;
				c=false;
				d=false;
				for(j=0;j<=num_areas;j++)
				{
					if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
					{
						if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
							(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
						{	
							if(areas[j].a==areas[i].a ||
								areas[j].b==areas[i].a ||
								areas[j].c==areas[i].a ||
								areas[j].d==areas[i].a)
								a=true;
							if(areas[j].a==areas[i].b ||
								areas[j].b==areas[i].b ||
								areas[j].c==areas[i].b ||
								areas[j].d==areas[i].b)
								b=true;
							if(areas[j].a==areas[i].c ||
								areas[j].b==areas[i].c ||
								areas[j].c==areas[i].c ||
								areas[j].d==areas[i].c)
								c=true;
							if(areas[j].a==areas[i].d ||
								areas[j].b==areas[i].d ||
								areas[j].c==areas[i].d ||
								areas[j].d==areas[i].d)
								d=true;

						}
					}
				}

				//corner areas will have 1 bool (out of a-c) false only
				if((!a && b && c && d) ||
					(a && !b && c && d) ||
					(a && b && !c && d) ||
					(a && b && c && !d) ||
					(!a && !b && c && d) ||
					(a && b && !c && !d) ||
					(a && !b && !c && d) ||
					(!a && b && c && !d))
				{
					//Vector aa,bb;
					//now find all the areas we can merge this one with
					bool merged;
					merged=true;
					while(merged)
					{
						stk_cnt=0;
						merged=false;
						if((areas[i].d.x-areas[i].a.x)>
							(areas[i].b.y-areas[i].a.y))
						{
							//x>y so expand in the y direction
							for(j=0;j<=num_areas;j++)
							{
								if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
								{
									if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
										(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
									{	
										if(!a || !d)
										{
											if(areas[j].a==areas[i].b &&
												areas[j].d==areas[i].c)
											{
												areas[i].b=areas[j].b;
												areas[i].c=areas[j].c;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].a==areas[i].b &&
													(areas[j].d.x<areas[i].c.x))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].d.x<areas[i].c.x) &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].d==areas[i].c &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														areas[i].b=areas[stk[0]].b;
														areas[i].c=areas[j].c;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
										if(!b || !c)
										{
											if(areas[j].b==areas[i].a &&
												areas[j].c==areas[i].d)
											{
												areas[i].a=areas[j].a;
												areas[i].d=areas[j].d;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].b==areas[i].a &&
													(areas[j].c.x<areas[i].d.x))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].c.x<areas[i].d.x) &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].c==areas[i].d &&
														(areas[j].a==areas[stk[stk_cnt-1]].d) &&
														(areas[j].b==areas[stk[stk_cnt-1]].c))
													{
														areas[i].a=areas[stk[0]].a;
														areas[i].d=areas[j].d;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
									}
								}
							}
						}
						else
						{
							//x<=y so expand in the x direction
							for(j=0;j<=num_areas;j++)
							{
								if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
								{
									if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
										(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
									{	
										if(!a || !b)
										{
											if(areas[j].a==areas[i].d &&
												areas[j].b==areas[i].c)
											{
												areas[i].d=areas[j].d;
												areas[i].c=areas[j].c;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].a==areas[i].d &&
													(areas[j].b.y<areas[i].c.y))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].b.y<areas[i].c.y) &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].b==areas[i].c &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														areas[i].d=areas[stk[0]].d;
														areas[i].c=areas[j].c;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
										if(!d || !c)
										{
											if(areas[j].d==areas[i].a &&
												areas[j].c==areas[i].b)
											{
												areas[i].a=areas[j].a;
												areas[i].b=areas[j].b;
												areas[j].flags=W_FL_DELETED;
												merged=true;
												j=num_areas;
											}
											//else to multiple merge
											else
											{
												//start
												if(stk_cnt==0 && 
													areas[j].d==areas[i].a &&
													(areas[j].c.y<areas[i].b.y))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												else if(stk_cnt!=0)
												{
													//middle
													if(stk_cnt<stk_sz &&
														(areas[j].c.y<areas[i].b.y) &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														stk[stk_cnt]=j;
														stk_cnt++;
														j=-1;
													}
													//end
													if(areas[j].c==areas[i].b &&
														(areas[j].a==areas[stk[stk_cnt-1]].b) &&
														(areas[j].d==areas[stk[stk_cnt-1]].c))
													{
														areas[i].a=areas[stk[0]].a;
														areas[i].b=areas[j].b;
														areas[j].flags=W_FL_DELETED;
														for(k=0;k<stk_cnt;k++)
														{
															areas[stk[k]].flags=W_FL_DELETED;
														}
														merged=true;
														j=num_areas;
													}
												}
											}
										}
									}
								}
							}
						}
					}
					//end merged
				}

			}
		}
	}
	//and the final lot?
	for(i=0;i<=num_areas;i++)
	{
		if (!((areas[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2 &&
				(areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4))
			{
				a=false;
				b=false;
				c=false;
				d=false;
				{
					//Vector aa,bb;
					//now find all the areas we can merge this one with
					bool merged;
					merged=true;
					while(merged)
					{
						stk_cnt=0;
						merged=false;
						//x>y so expand in the y direction
						for(j=0;j<=num_areas;j++)
						{
							if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
							{
								if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
									(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
								{	
									if(!a || !d)
									{
										if(areas[j].a==areas[i].b &&
											areas[j].d==areas[i].c)
										{
											areas[i].b=areas[j].b;
											areas[i].c=areas[j].c;
											areas[j].flags=W_FL_DELETED;
											merged=true;
											j=num_areas;
										}
										//else to multiple merge
										else
										{
											//start
											if(stk_cnt==0 && 
												areas[j].a==areas[i].b &&
												(areas[j].d.x<areas[i].c.x))
											{
												stk[stk_cnt]=j;
												stk_cnt++;
												j=-1;
											}
											else if(stk_cnt!=0)
											{
												//middle
												if(stk_cnt<stk_sz &&
													(areas[j].d.x<areas[i].c.x) &&
													(areas[j].a==areas[stk[stk_cnt-1]].d) &&
													(areas[j].b==areas[stk[stk_cnt-1]].c))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												//end
												if(areas[j].d==areas[i].c &&
													(areas[j].a==areas[stk[stk_cnt-1]].d) &&
													(areas[j].b==areas[stk[stk_cnt-1]].c))
												{
													areas[i].b=areas[stk[0]].b;
													areas[i].c=areas[j].c;
													areas[j].flags=W_FL_DELETED;
													for(k=0;k<stk_cnt;k++)
													{
														areas[stk[k]].flags=W_FL_DELETED;
													}
													merged=true;
													j=num_areas;
												}
											}
										}
									}
									if(!b || !c)
									{
										if(areas[j].b==areas[i].a &&
											areas[j].c==areas[i].d)
										{
											areas[i].a=areas[j].a;
											areas[i].d=areas[j].d;
											areas[j].flags=W_FL_DELETED;
											merged=true;
											j=num_areas;
										}
										//else to multiple merge
										else
										{
											//start
											if(stk_cnt==0 && 
												areas[j].b==areas[i].a &&
												(areas[j].c.x<areas[i].d.x))
											{
												stk[stk_cnt]=j;
												stk_cnt++;
												j=-1;
											}
											else if(stk_cnt!=0)
											{
												//middle
												if(stk_cnt<stk_sz &&
													(areas[j].c.x<areas[i].d.x) &&
													(areas[j].a==areas[stk[stk_cnt-1]].d) &&
													(areas[j].b==areas[stk[stk_cnt-1]].c))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												//end
												if(areas[j].c==areas[i].d &&
													(areas[j].a==areas[stk[stk_cnt-1]].d) &&
													(areas[j].b==areas[stk[stk_cnt-1]].c))
												{
													areas[i].a=areas[stk[0]].a;
													areas[i].d=areas[j].d;
													areas[j].flags=W_FL_DELETED;
													for(k=0;k<stk_cnt;k++)
													{
														areas[stk[k]].flags=W_FL_DELETED;
													}
													merged=true;
													j=num_areas;
												}
											}
										}
									}
								}
							}
						}
						//x<=y so expand in the x direction
						for(j=0;j<=num_areas;j++)
						{
							if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
							{
								if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
									(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
								{	
									if(!a || !b)
									{
										if(areas[j].a==areas[i].d &&
											areas[j].b==areas[i].c)
										{
											areas[i].d=areas[j].d;
											areas[i].c=areas[j].c;
											areas[j].flags=W_FL_DELETED;
											merged=true;
											j=num_areas;
										}
										//else to multiple merge
										else
										{
											//start
											if(stk_cnt==0 && 
												areas[j].a==areas[i].d &&
												(areas[j].b.y<areas[i].c.y))
											{
												stk[stk_cnt]=j;
												stk_cnt++;
												j=-1;
											}
											else if(stk_cnt!=0)
											{
												//middle
												if(stk_cnt<stk_sz &&
													(areas[j].b.y<areas[i].c.y) &&
													(areas[j].a==areas[stk[stk_cnt-1]].b) &&
													(areas[j].d==areas[stk[stk_cnt-1]].c))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												//end
												if(areas[j].b==areas[i].c &&
													(areas[j].a==areas[stk[stk_cnt-1]].b) &&
													(areas[j].d==areas[stk[stk_cnt-1]].c))
												{
													areas[i].d=areas[stk[0]].d;
													areas[i].c=areas[j].c;
													areas[j].flags=W_FL_DELETED;
													for(k=0;k<stk_cnt;k++)
													{
														areas[stk[k]].flags=W_FL_DELETED;
													}
													merged=true;
													j=num_areas;
												}
											}
										}
									}
									if(!d || !c)
									{
										if(areas[j].d==areas[i].a &&
											areas[j].c==areas[i].b)
										{
											areas[i].a=areas[j].a;
											areas[i].b=areas[j].b;
											areas[j].flags=W_FL_DELETED;
											merged=true;
											j=num_areas;
										}
										//else to multiple merge
										else
										{
											//start
											if(stk_cnt==0 && 
												areas[j].d==areas[i].a &&
												(areas[j].c.y<areas[i].b.y))
											{
												stk[stk_cnt]=j;
												stk_cnt++;
												j=-1;
											}
											else if(stk_cnt!=0)
											{
												//middle
												if(stk_cnt<stk_sz &&
													(areas[j].c.y<areas[i].b.y) &&
													(areas[j].a==areas[stk[stk_cnt-1]].b) &&
													(areas[j].d==areas[stk[stk_cnt-1]].c))
												{
													stk[stk_cnt]=j;
													stk_cnt++;
													j=-1;
												}
												//end
												if(areas[j].c==areas[i].b &&
													(areas[j].a==areas[stk[stk_cnt-1]].b) &&
													(areas[j].d==areas[stk[stk_cnt-1]].c))
												{
													areas[i].a=areas[stk[0]].a;
													areas[i].b=areas[j].b;
													areas[j].flags=W_FL_DELETED;
													for(k=0;k<stk_cnt;k++)
													{
														areas[stk[k]].flags=W_FL_DELETED;
													}
													merged=true;
													j=num_areas;
												}
											}
										}
									}
								}
							}
						}
					}
					//end merged
				}

			}
		}
	}


	//first group areas that are slopes/stairs

	//given a area, slope in either direction, aslong as
	//a)size of adjacent area matches
	//b)step is the same (more than 2 areas)
	//c)same direction as first ones found
	for(i=0;i<=num_areas;i++)
	{
		if (!((areas[i].flags & W_FL_DELETED)==W_FL_DELETED))
		{
			if (((areas[i].flags & A_FL_1) == A_FL_1 && (areas[i].flags & A_FL_2) == A_FL_2 &&
				(areas[i].flags & A_FL_3) == A_FL_3 && (areas[i].flags & A_FL_4) == A_FL_4))
			{
				float sx,sy;
				sx=areas[i].d.x-areas[i].a.x;
				sy=areas[i].b.y-areas[i].a.y;
				for(j=0;j<=num_areas;j++)
				{
					if (!((areas[j].flags & W_FL_DELETED)==W_FL_DELETED) && i!=j)
					{
						if (((areas[j].flags & A_FL_1) == A_FL_1 && (areas[j].flags & A_FL_2) == A_FL_2 &&
							(areas[j].flags & A_FL_3) == A_FL_3 && (areas[j].flags & A_FL_4) == A_FL_4))
						{	
							//find neighbours to i
							float z;

							z=areas[j].a.z-areas[i].d.z;
							if(areas[j].b.z-areas[i].c.z!=z) 
								z=0;
							if(areas[j].a.x==areas[i].d.x && areas[j].a.y==areas[i].d.y &&
								areas[j].b.x==areas[i].c.x && areas[j].b.y==areas[i].c.y &&
								//areas[j].a.z==areas[j].d.z && areas[j].b.z==areas[j].c.z &&
								areas[j].a.z==areas[j].b.z && areas[j].d.z==areas[j].c.z &&
								(z!=0 && (z>=-32 && z<=32)))
							{
								//now check the size
								//if it matches, merge
								if(sx==areas[j].d.x-areas[j].a.x &&
									sy==areas[j].b.y-areas[j].a.y)
								{
									areas[i].d=areas[j].d;
									areas[i].c=areas[j].c;
									areas[j].flags=W_FL_DELETED;

									//j=-1;
									//now search for neighbours with the same z
									//and same size
									for(k=0;k<=num_areas;k++)
									{
										if (!((areas[k].flags & W_FL_DELETED)==W_FL_DELETED) && j!=k && i!=k)
										{
											if (((areas[k].flags & A_FL_1) == A_FL_1 && (areas[k].flags & A_FL_2) == A_FL_2 &&
												(areas[k].flags & A_FL_3) == A_FL_3 && (areas[k].flags & A_FL_4) == A_FL_4))
											{	
												float zz;
												zz=areas[k].a.z-areas[i].d.z;
												if(areas[k].b.z-areas[i].c.z!=zz) 
													zz=0;
												if(areas[k].a.x==areas[i].d.x && areas[k].a.y==areas[i].d.y &&
													areas[k].b.x==areas[i].c.x && areas[k].b.y==areas[i].c.y &&
													//areas[k].a.z==areas[k].d.z && areas[k].b.z==areas[k].c.z &&
													areas[k].a.z==areas[k].b.z && areas[k].d.z==areas[k].c.z &&
													//zz!=0 && (z==zz || z==-zz || -z==zz))
													(zz!=0 && (zz>=-32 && zz<=32)))
												{
													//now check the size
													//if it matches, merge
													if(sx==areas[k].d.x-areas[k].a.x &&
														sy==areas[k].b.y-areas[k].a.y)
													{
														areas[i].d=areas[k].d;
														areas[i].c=areas[k].c;
														areas[k].flags=W_FL_DELETED;
														k=-1;
													}				  
												}
												zz=areas[k].d.z-areas[i].a.z;
												if(areas[k].c.z-areas[i].b.z!=zz) 
													zz=0;
												if(areas[k].d.x==areas[i].a.x && areas[k].d.y==areas[i].a.y &&
													areas[k].c.x==areas[i].b.x && areas[k].c.y==areas[i].b.y &&
													//areas[k].a.z==areas[k].d.z && areas[k].b.z==areas[k].c.z &&
													areas[k].a.z==areas[k].b.z && areas[k].d.z==areas[k].c.z &&
													//zz!=0 && (z==zz || z==-zz || -z==zz))
													(zz!=0 && (zz>=-32 && zz<=32)))
												{
													//now check the size
													//if it matches, merge
													if(sx==areas[k].d.x-areas[k].a.x &&
														sy==areas[k].b.y-areas[k].a.y)
													{
														areas[i].a=areas[k].a;
														areas[i].b=areas[k].b;
														areas[k].flags=W_FL_DELETED;
														k=-1;
													}				  
												}
											}
										}
									}
								}				  
							}
							//other way round
							z=areas[j].a.z-areas[i].b.z;
							if(areas[j].d.z-areas[i].c.z!=z) 
								z=0;
							if(areas[j].a.x==areas[i].b.x && areas[j].a.y==areas[i].b.y &&
								areas[j].d.x==areas[i].c.x && areas[j].d.y==areas[i].c.y &&
								//areas[j].a.z==areas[j].d.z && areas[j].b.z==areas[j].c.z &&
								areas[j].a.z==areas[j].b.z && areas[j].d.z==areas[j].c.z &&
								(z!=0 && (z>=-32 && z<=32)))
							{
								//now check the size
								//if it matches, merge
								if(sx==areas[j].d.x-areas[j].a.x &&
									sy==areas[j].b.y-areas[j].a.y)
								{
									areas[i].b=areas[j].b;
									areas[i].c=areas[j].c;
									areas[j].flags=W_FL_DELETED;

									//j=-1;
									//now search for neighbours with the same z
									//and same size
									for(k=0;k<=num_areas;k++)
									{
										if (!((areas[k].flags & W_FL_DELETED)==W_FL_DELETED) && j!=k && i!=k)
										{
											if (((areas[k].flags & A_FL_1) == A_FL_1 && (areas[k].flags & A_FL_2) == A_FL_2 &&
												(areas[k].flags & A_FL_3) == A_FL_3 && (areas[k].flags & A_FL_4) == A_FL_4))
											{	
												float zz;
												zz=areas[k].a.z-areas[i].d.z;
												if(areas[k].b.z-areas[i].c.z!=zz) 
													zz=0;
												if(areas[k].a.x==areas[i].b.x && areas[k].a.y==areas[i].b.y &&
													areas[k].d.x==areas[i].c.x && areas[k].d.y==areas[i].c.y &&
													//areas[k].a.z==areas[k].d.z && areas[k].b.z==areas[k].c.z &&
													areas[k].a.z==areas[k].b.z && areas[k].d.z==areas[k].c.z &&
													//zz!=0 && (z==zz || z==-zz || -z==zz))
													(zz!=0 && (zz>=-32 && zz<=32)))
												{
													//now check the size
													//if it matches, merge
													if(sx==areas[k].d.x-areas[k].a.x &&
														sy==areas[k].b.y-areas[k].a.y)
													{
														areas[i].b=areas[k].b;
														areas[i].c=areas[k].c;
														areas[k].flags=W_FL_DELETED;
														k=-1;
													}				  
												}
												zz=areas[k].d.z-areas[i].a.z;
												if(areas[k].c.z-areas[i].b.z!=zz) 
													zz=0;
												if(areas[k].b.x==areas[i].a.x && areas[k].b.y==areas[i].a.y &&
													areas[k].c.x==areas[i].d.x && areas[k].c.y==areas[i].d.y &&
													//areas[k].a.z==areas[k].d.z && areas[k].b.z==areas[k].c.z &&
													areas[k].a.z==areas[k].b.z && areas[k].d.z==areas[k].c.z &&
													//zz!=0 && (z==zz || z==-zz || -z==zz))
													(zz!=0 && (zz>=-32 && zz<=32)))
												{
													//now check the size
													//if it matches, merge
													if(sx==areas[k].d.x-areas[k].a.x &&
														sy==areas[k].b.y-areas[k].a.y)
													{
														areas[i].a=areas[k].a;
														areas[i].d=areas[k].d;
														areas[k].flags=W_FL_DELETED;
														k=-1;
													}				  
												}
											}
										}
									}
								}				  
							}
						}
					}

				}

			}
		}
	}

}

void JunctionCalculate()
{
	return;
	int i,j;
	int a[MAX_WAYPOINTS];
	for (i=0; i < num_waypoints; i++)
	{
		a[i]=0;
	}
	for (i=0; i < num_waypoints; i++)
	{
		for (j=0; j < num_waypoints; j++)
		{
			if(i!=j)
			{
				TraceResult tr;
				UTIL_TraceLine(waypoints[i].origin,
					waypoints[j].origin,
					ignore_monsters,ignore_glass,NULL, &tr );
				if(tr.flFraction >= 1.0)
					a[i]++;
			}
		}
	}
	PATH *p;
	int count;
	for (i=0; i < num_waypoints; i++)
	{
		p=paths[i];
		count=0;
		while (p != NULL)
		{
			j = 0;

			while (j < MAX_PATH_INDEX)
			{
				if (p->index[j] != -1)
			 {
				 if(a[j]>count)
					 count=a[j];
			 }

			 j++;
			}
			p = p->next;  // go to next node in linked list
		}
		//now do the bit to work out whats a junction or not
		if(a[i]>count)
			is_junction[i]=true;
	}
}

int JunctionGoto(bot_t *pBot,int from,int to)
{

	if(to==from) return -1; //..........
	int team;
	team=UTIL_GetTeam(pBot->pEdict);
	if(pBot->next_junc==to && pBot->next_junc!=-1) //when next junc is goal...
	{
		pBot->last_distance=0;
		return WaypointRouteFromTo(from,to,team);
	}
	if(pBot->bot_skill>=4) //to stupid..
	{
		pBot->last_distance=0;
		return WaypointRouteFromTo(from,to,team);
	}
	if(pBot->next_junc!=from && pBot->next_junc!=-1) //not reached next step..
	{
		return WaypointRouteFromTo(from,pBot->next_junc,team);
	}

	//for the multipath stuff, gotta calc the data if we have a new start/end
	//here.....
	//did think of doing it in the distance return func, but wouldn't work well
	//cus the way this function works.

	if(pBot->last_dest!=to)
	{
		pBot->last_dest=to;
		//WaypointToms(pBot->multi_path_dist,team,from,to);
	}
	//char s[255];
	//sprintf(s,"=%d %d\n",pBot->last_dest,to);
	//ALERT(at_console, s);

	int index,i;
	index=-1;
	int from_d[MAX_WAYPOINTS];
	int to_d[MAX_WAYPOINTS];

	int from_min,from_max,to_min,to_max;
	int from_min_i,from_max_i,to_min_i,to_max_i;
	from_min=99999;
	to_min=99999;
	from_max=0;
	to_max=0;
	from_min_i=-1;
	to_min_i=-1;
	from_max_i=-1;
	to_max_i=-1;

	int to_min_r,to_max_r,to_r_c,to_r_a;
	to_min_r=99999;
	to_max_r=0;
	to_r_c=0;
	to_r_a=0;
	//order all junctions in distance to player..
	//and all junctions in distance to goal..

	//dont really need to sort them.. 
	//as it'll mess up the order (which point it belongs to)
	//instead.. rate each one as you go..kinda like find min..
	//but with some randomness to it.


	char msg[255];
	int c1,c2;
	c1=0;
	c2=0;

	for(i=0;i<num_waypoints;i++)
	{
		//check that it is a junction..
		from_d[i]=-1;
		to_d[i]=-1;
		if(is_junction[i])
		{
			//discount the current waypoint
			if(from!=i && i!=pBot->last_wp[0] && i!=pBot->last_wp[1] &&
				i!=pBot->last_wp[2] && i!=pBot->last_wp[3] && 
				i!=pBot->last_wp[4]) 
				// && i!=pBot->last_wp[5]
			{
				from_d[i]=WaypointMultiDistanceFromTo(pBot,from,i,team);
				//{ fp=fopen("bot.txt","a"); fprintf(fp,"-- s %d d %d dis %d\n",from,i,from_d[i]); fclose(fp); }
				if(from_d[i]!=-1)
				{
					//debugging
					if(pBot->last_distance<from_d[i]) 
					{
						WaypointDrawBeam(INDEXENT(1), pBot->pEdict->v.origin,waypoints[i].origin, 10, 2, 0, 250, 0, 200, 10);
						sprintf(msg,"+ %d\n",from_d[i]);
						ALERT(at_console, msg);
						c1++;
					}
					else if(pBot->last_distance>=from_d[i]) 
					{
						WaypointDrawBeam(INDEXENT(1), pBot->pEdict->v.origin,waypoints[i].origin, 10, 2, 250, 0, 0, 200, 10);
						sprintf(msg,"- %d\n",from_d[i]);
						ALERT(at_console, msg);
					}
					if(from_d[i]>0) c2++;



					if(from_d[i]>from_max)
					{
						from_max=from_d[i];
						from_max_i=i;
					}
					if(from_d[i]<from_min)
					{
						from_min=from_d[i];
						from_min_i=i;
					}
				}
				to_d[i]=WaypointMultiDistanceFromTo(pBot,i,to,team);
				if(to_d[i]!=-1)
				{
					if(to_d[i]>to_max)
					{
						to_max=to_d[i];
						to_max_i=i;
					}
					if(to_d[i]<to_min)
					{
						to_min=to_d[i];
						to_min_i=i;
					}
				}
				if(WaypointDistanceFromTo(from,i,team)<400 && from_d[i]!=-1 && to_d[i]!=-1)
				{
					if(to_d[i]>to_max_r)
						to_max_r=to_d[i];
					if(to_d[i]<to_min_r)
						to_min_r=to_d[i];
					to_r_c++;
					to_r_a=to_r_a+to_d[i];
				}
			}
		}
	}
	if(to_r_c>0) to_r_a=abs(to_r_a/to_r_c);


	//debug
	if(c1>1) UTIL_HostSay(pBot->pEdict, 0, "more than one choice for me!");
	if(c2>1) UTIL_HostSay(pBot->pEdict, 0, "more than one path");

	int dis,ran,count,to_dd;
	float rate;
	count=0;
	while(index==-1 && count<11)
	{
		//ran=RANDOM_LONG(-500,500-(count*50));
		dis=pBot->last_distance;
		//if rate <0 more likely to choose route..
		//if rate >0 less likely..
		//rate is only +/- 5
		rate=0;
		//if(dis>200) //this would be distance to the goal!
		//{
		for(i=0;i<num_waypoints;i++)
		{
			// && to_d[i]!=-1
			if(is_junction[i] && from_d[i]!=-1)
			{
				//discount the current waypoint
				if(from!=i && i!=pBot->last_wp[0] && i!=pBot->last_wp[1] &&
					i!=pBot->last_wp[2] && i!=pBot->last_wp[3] && 
					i!=pBot->last_wp[4]) 
					// && i!=pBot->last_wp[5]
				{
					//----------------------
					//if(((to_d[i]-RANDOM_LONG(0,to_min+from_min))+((from_d[i])+ran)<=dis))
					//if((from_d[i]+ran)<=from_min && (to_d[i]-RANDOM_LONG(0,to_min+from_min+ran)<=dis))
					ran=RANDOM_FLOAT(-5+(count/2),5);
					//instead of from_d[i]<400 we do the measurements manually
					//WaypointDistanceFromTo(from,i,team)<(400+(100*(count/2))) && 
					// && WaypointDistanceFromTo(from,i,team)!=-1
					if(ran>=rate)
					{
						to_dd=from_d[i];
						//if its an averagely far away point...hmm
						//may cause backing up..
						//if(to_dd==to_max_r && (to_max_r-to_max_r)>RANDOM_LONG(600,800))
						//{
						//	to_dd=to_r_a;
						//}
						//if this point is oneway(i.e. drop into water)
						//and thus is further away from your goal than  other near points
						//reduce it.
						//if(to_dd>to_min_r+700) 
						//{
						//	to_dd=(to_dd-(to_max_r-to_min_r))+RANDOM_LONG(0,abs(to_max_r/2));
						//}
						//don't like stuff thats <0
						//if(to_dd<0) to_dd=to_d[i];
						if(to_dd>dis)
						{
							//why rate=ran?
							//ah, so this is the only choice..heh
							if(RANDOM_LONG(0,10)<4) rate=ran;
							index=i;
							dis=from_d[i];
							if(dis<0) dis=0;
						}
						//{ fp=fopen("bot.txt","a"); fprintf(fp,"-- s %d d %d dis %d index %d\n",from,i,from_d[i],index); fclose(fp); }
					}
				}
			}
		}
		//}
		count++;
	}
	if(index!=-1) 
	{
		pBot->last_distance=dis;
		pBot->next_junc=index; //this is our next step destination..
		pBot->last_wp[4]=pBot->last_wp[3];
		pBot->last_wp[2]=pBot->last_wp[1];
		pBot->last_wp[1]=pBot->last_wp[0];
		pBot->last_wp[0]=index;
	}
	return index;
}

void ProcessCommanderList(void)
{
	// ProcessCommanderList - This command will read in a list of commanders from a text file. The id's in this file will be
	// used to determine players that have access to command the bots, as in assign roles, classes positions...
	char msg[255];
	char buffer[80];
	char filename[255];
	//// delete dynamic memory
	//LIter<char *> iter(&commanders);
	//for (iter.begin(); !iter.end(); ++iter)
	//{	
	//}
	commanders.clear();
	char invalidChars[] = " abcdefghijklmnopqrstuvwxyz,./<>?;'\"[]{}-=+!@#$%^&*()";
	
	FILE *inFile = NULL;
	UTIL_BuildFileName(filename, "foxbot_commanders.txt", NULL);
	inFile = fopen(filename, "r");

	if (inFile)
	{
		if (IS_DEDICATED_SERVER())	
			printf("[Config] Reading foxbot_commanders.txt\n");	
		else
		{
			sprintf(msg,"[Config] Reading foxbot_commanders.txt\n");
			ALERT( at_console, msg);
		}
	} else
	{
		if (IS_DEDICATED_SERVER())	
			printf("[Config] Error Reading foxbot_commanders.txt\n");	
		else
		{
			sprintf(msg,"[Config] Error Reading foxbot_commanders.txt\n");
			ALERT( at_console, msg);
		}
		return;
	}
	
	// Read the file.
	while (fgets( buffer, 80, inFile ))
	{		
		// Skip lines with comments
		if ((int)strlen(buffer) > 2)
		{
			if (buffer[0] == '/' && buffer[1] == '/')
				continue;
		}
		bool valid = true;
		// Search for invalid characters.
		for (int i = 0; i < (int)strlen(buffer); i++)
		{			
			for (int j = 0; j < (int)strlen(invalidChars); j++)
			{
				char ch = invalidChars[j];

				if (strchr(buffer, ch))
				{
					valid = false;
					if (IS_DEDICATED_SERVER())	
						printf("[Config] foxbot_commanders.txt : Invalid Character %c\n", ch);	
					else
					{
						sprintf(msg,"[Config] foxbot_commanders.txt : Invalid Character %c\n", ch);
						ALERT( at_console, msg);
					}
				}
			}			
		} 
		// It's good.
		if (valid)
		{		
			char *uId = new char[80];
			strcpy(uId, buffer);

			// Get rid of line feeds
			if (uId[strlen(uId)-1] == '\n' || uId[strlen(uId)-1] == '\r' || uId[strlen(uId)-1] == EOF)
			{
				uId[strlen(uId)-1] = NULL;
			}

			{ fp=fopen("bot.txt","a"); fprintf(fp,"LOAD USERID: %s\n",uId); fclose(fp); }
			commanders.addTail(uId);
			if (IS_DEDICATED_SERVER())	
				printf("[Config] foxbot_commanders.txt : Loaded User %s\n", buffer);	
			else
			{
				sprintf(msg,"[Config] foxbot_commanders.txt : Loaded User %s\n", buffer);
				ALERT( at_console, msg);
			}
		}
	}
	// Report status
	if (IS_DEDICATED_SERVER())	
		printf("[Config] foxbot_commanders.txt : Loaded %d users\n", commanders.size());	
	else
	{
		sprintf(msg,"[Config] foxbot_commanders.txt : Loaded %d users\n", commanders.size());
		ALERT( at_console, msg);
	}
	fclose(inFile);
}
