const { app, BrowserWindow, ipcMain, dialog } = require("electron");
const os = require("os");
const path = require("path");
const fs = require("fs");
const { exec } = require("child_process");
const https = require("https");

const logFile = path.join(__dirname, "error.log");

let mainWindow;
let isQuitting = false;

// Get the directory where the executable is located for portable storage
function getPortableDataDir() {
  // In development, use __dirname
  // In packaged app, use the directory containing the executable
  if (app.isPackaged) {
    return path.dirname(process.execPath);
  } else {
    return __dirname;
  }
}

const portableDataDir = getPortableDataDir();
const selectedPrintersFile = path.join(
  portableDataDir,
  "selectedPrinters.json"
);
const machineIdFile = path.join(portableDataDir, "machineId.json");
const settingsFile = path.join(portableDataDir, "settings.json");

// Printer name lists
const labelsPrinters = [
  "NLWRTNP02003",
  "NLWRTNP02004",
  "NLWRTNP02005",
  "NLWRTNP02006",
  "NLWRTNP02007",
  "NLWRTNP02008",
  "NLWRTNP02009",
  "NLWRTNP02010",
  "NLWRTNP02023",
  "NLWRTNP02024",
  "NLWRTNP02025",
  "NLWRTNP02026",
  "NLWRTNP02027",
  "NLWRTNP02028",
  "NLWRTNP03018",
  "NLWRTNP03019",
  "NLWRTNP03020",
  "NLWRTNP04023",
  "NLWRTNP04024",
  "NLWRTNP04025",
  "NLWRTNP04026",
  "NLWRTNP04027",
  "NLWRTNP04028",
  "NLWRTNP04029",
  "NLWRTNP04030",
  "NLWRTNP04031",
  "NLWRTNP04032",
  "NLWRTNP04033",
  "NLWRTNP04034",
  "NLWRTNP04035",
  "NLWRTNP04036",
  "NLWRTNP04037",
  "NLWRTNP04038",
  "NLWRTNP04039",
  "NLWRTNP04040",
  "NLWRTNP04041",
  "NLWRTNP04042",
  "NLWRTNP04044",
  "NLWRTNP04045",
  "NLWRTNP04046",
  "NLWRTNP04047",
  "NLWRTNP04048",
  "NLWRTNP04051",
  "NLWRTNP04052",
  "NLWRTNP04057",
  "NLWRTNP09015",
  "NLWRTNP09040",
];
const colorPrinters = [
  "NLWRTNP02000",
  "NLWRTNP02001",
  "NLWRTNP02002",
  "NLWRTNP03013",
  "NLWRTNP03014",
  "NLWRTNP03015",
  "NLWRTNP03016",
  "NLWRTNP03017",
  "NLWRTNP03022",
  "NLWRTNP04016",
  "NLWRTNP04017",
  "NLWRTNP04018",
  "NLWRTNP04019",
  "NLWRTNP04020",
  "NLWRTNP04021",
  "NLWRTNP04022",
  "NLWRTNP04050",
];
const blackPrinters = [
  "NLWRTNP02012",
  "NLWRTNP02020",
  "NLWRTNP02021",
  "NLWRTNP02022",
  "NLWRTNP03001",
  "NLWRTNP03002",
  "NLWRTNP03003",
  "NLWRTNP03004",
  "NLWRTNP03005",
  "NLWRTNP03006",
  "NLWRTNP03007",
  "NLWRTNP03008",
  "NLWRTNP03009",
  "NLWRTNP03010",
  "NLWRTNP03011",
  "NLWRTNP03012",
  "NLWRTNP03023",
  "NLWRTNP04001",
  "NLWRTNP04002",
  "NLWRTNP04003",
  "NLWRTNP04004",
  "NLWRTNP04005",
  "NLWRTNP04006",
  "NLWRTNP04007",
  "NLWRTNP04008",
  "NLWRTNP04009",
  "NLWRTNP04010",
  "NLWRTNP04011",
  "NLWRTNP04012",
  "NLWRTNP04013",
  "NLWRTNP04014",
  "NLWRTNP04015",
];

// Function to determine the icon file based on the printer name
function getIconForPrinter(printerName) {
  const iconsPath = path.join(__dirname, "Icons");

  // Helper function to clean and normalize printer names
  const normalizePrinterName = (name) => name.split(" (")[0].trim(); // Remove anything after " (" and trim spaces

  // Normalize the input name
  const normalizedPrinterName = normalizePrinterName(printerName);

  // Check for matches in the lists
  if (labelsPrinters.includes(normalizedPrinterName)) {
    return path.join(iconsPath, "labels.ico");
  }
  if (colorPrinters.includes(normalizedPrinterName)) {
    return path.join(iconsPath, "color.ico");
  }
  if (blackPrinters.includes(normalizedPrinterName)) {
    return path.join(iconsPath, "black.ico");
  }

  // Default icon
  return path.join(iconsPath, "default.ico");
}

// Function to reset printer data if the machine is different
function resetPrinterDataIfNeeded() {
  const machineId = getMachineId();
  if (fs.existsSync(machineIdFile)) {
    const savedId = JSON.parse(
      fs.readFileSync(machineIdFile, "utf8")
    ).machineId;
    if (savedId !== machineId) {
      console.log("Machine ID has changed. Resetting printer data.");
      if (fs.existsSync(selectedPrintersFile)) {
        fs.unlinkSync(selectedPrintersFile); // Delete saved selected printers
      }
    }
  }
  // Save the new machine ID
  fs.writeFileSync(machineIdFile, JSON.stringify({ machineId }, null, 2));
}

// Function to get the machine ID (hostname)
function getMachineId() {
  return os.hostname();
}

// Function to read saved selected printers
function readSelectedPrinters() {
  if (fs.existsSync(selectedPrintersFile)) {
    return JSON.parse(fs.readFileSync(selectedPrintersFile, "utf8"));
  }
  return [];
}

// Function to save selected printers
function saveSelectedPrinters(selectedPrinters) {
  fs.writeFileSync(
    selectedPrintersFile,
    JSON.stringify(selectedPrinters, null, 2)
  );
}

// Function to validate desktop shortcuts
function validateDesktopShortcuts(selectedPrinters) {
  const settings = readSettings();
  // Only validate desktop shortcuts if desktop shortcut creation is enabled
  if (!settings.createDesktopShortcut) {
    return selectedPrinters; // Return all printers if desktop shortcuts are disabled
  }

  const desktopPath = app.getPath("desktop");
  return selectedPrinters.filter((printer) => {
    const shortcutPath = path.join(desktopPath, `${printer}.lnk`);
    return fs.existsSync(shortcutPath); // Keep only printers with existing shortcuts
  });
}

function saveSettings(settings) {
  console.log("Saving settings to:", settingsFile);
  console.log("Settings being saved:", settings);
  fs.writeFileSync(settingsFile, JSON.stringify(settings, null, 2));
  console.log("Settings saved successfully");
}

// Update the readSettings function
function readSettings() {
  console.log("Reading settings from:", settingsFile);
  try {
    if (fs.existsSync(settingsFile)) {
      const settings = JSON.parse(fs.readFileSync(settingsFile, "utf8"));
      console.log("Settings loaded:", settings);
      return {
        filterType: settings.filterType || "all",
        sortOption: settings.sortOption || "a-z",
        searchTerm: settings.searchTerm || "",
        theme: settings.theme || "dark",
        createDesktopShortcut:
          settings.createDesktopShortcut !== undefined
            ? settings.createDesktopShortcut
            : true,
        createTaskbarShortcut:
          settings.createTaskbarShortcut !== undefined
            ? settings.createTaskbarShortcut
            : true,
        firstRun: settings.firstRun !== undefined ? settings.firstRun : true,
      };
    }
  } catch (error) {
    console.error("Error reading settings:", error);
  }
  console.log("Using default settings (no settings file found)");
  return {
    filterType: "all",
    sortOption: "a-z",
    searchTerm: "",
    theme: "dark",
    createDesktopShortcut: true,
    createTaskbarShortcut: true,
    firstRun: true,
  };
}

function saveWindowState() {
  const currentSettings = readSettings();
  if (mainWindow.isMaximized()) {
    currentSettings.isMaximized = true;
  } else {
    const [width, height] = mainWindow.getSize();
    currentSettings.windowSize = { width, height };
    currentSettings.isMaximized = false;
  }
  // Save synchronously to ensure it's written
  fs.writeFileSync(settingsFile, JSON.stringify(currentSettings, null, 2));
}

// Add update check interval (4 hours)
const UPDATE_CHECK_INTERVAL = 4 * 60 * 60 * 1000;

// Update the checkForUpdates function
async function checkForUpdates(silent = false) {
  console.log("Checking for updates...", silent);
  const currentVersion = app.getVersion();
  const versionUrl =
    "https://download.teambunny.co/dhl/Programs/PrinterManager/updater/version-alternative.json";

  try {
    const response = await new Promise((resolve, reject) => {
      https
        .get(versionUrl, (res) => {
          let data = "";
          res.on("data", (chunk) => {
            data += chunk;
          });
          res.on("end", () => {
            try {
              const versionInfo = JSON.parse(data);
              resolve(versionInfo);
            } catch (err) {
              reject(err);
            }
          });
        })
        .on("error", reject);
    });

    if (response.version !== currentVersion) {
      const result = await dialog.showMessageBox(mainWindow, {
        type: "info",
        title: "Update Available",
        message: `A new version (${response.version}) is available!\n\nCurrent version: ${currentVersion}\n\nWhat's New:\n${response.releaseNotes}`,
        buttons: ["Download", "Later"],
        defaultId: 0,
      });

      if (result.response === 0) {
        // Download and run autoinstaller
        await downloadAndRunAutoInstaller();
      }
    } else if (!silent) {
      await dialog.showMessageBox(mainWindow, {
        type: "info",
        title: "DHL Printer Manager Alternative",
        message: "You're running the latest version!",
      });
    }
  } catch (error) {
    console.error("Update check error:", error);
    logError(error, "Error checking for updates");
    if (!silent) {
      await dialog.showMessageBox(mainWindow, {
        type: "error",
        title: "Update Check Failed",
        message: "Unable to check for updates. Please try again later.",
      });
    }
  }
}

// Function to download and run the autoinstaller
async function downloadAndRunAutoInstaller() {
  try {
    const https = require("https");
    const fs = require("fs");
    const path = require("path");
    const { exec } = require("child_process");
    const os = require("os");

    // Create temporary directory
    const tmpDir = path.join(os.tmpdir(), `dhl-update-${Date.now()}`);
    fs.mkdirSync(tmpDir, { recursive: true });

    const autoinstallUrl =
      "https://download.teambunny.co/dhl/Programs/PrinterManager/releases/Autoinstall-Alternative.ps1";
    const autoinstallPath = path.join(tmpDir, "Autoinstall-Alternative.ps1");

    console.log("Downloading autoinstaller to:", autoinstallPath);

    // Download the autoinstall script
    await new Promise((resolve, reject) => {
      const file = fs.createWriteStream(autoinstallPath);

      https
        .get(autoinstallUrl, (response) => {
          if (response.statusCode !== 200) {
            reject(
              new Error(`Download failed with status: ${response.statusCode}`)
            );
            return;
          }

          response.pipe(file);

          file.on("finish", () => {
            file.close();
            resolve();
          });

          file.on("error", (err) => {
            fs.unlinkSync(autoinstallPath);
            reject(err);
          });
        })
        .on("error", reject);
    });

    console.log("Autoinstaller downloaded successfully");

    // Show confirmation dialog
    const confirmResult = await dialog.showMessageBox(mainWindow, {
      type: "question",
      title: "Run Auto-Installer",
      message:
        "The auto-installer has been downloaded. This will:\n\n• Close this application\n• Download and install the latest version\n• Clean up old files and settings\n• Launch the new version\n\nDo you want to proceed?",
      buttons: ["Run Installer", "Cancel"],
      defaultId: 0,
    });

    if (confirmResult.response === 0) {
      // Create a batch script to run PowerShell and cleanup
      const batchPath = path.join(tmpDir, "run_update.bat");
      const batchContent = `@echo off
echo Starting DHL Printer Manager Alternative Auto-Installer...
cd /d "${tmpDir}"
powershell.exe -ExecutionPolicy Bypass -WindowStyle Normal -File "${autoinstallPath}"
echo Auto-installer completed.
pause
cd /d "%TEMP%"
rmdir /s /q "${tmpDir}" 2>nul
del "%~f0" 2>nul
`;

      fs.writeFileSync(batchPath, batchContent);

      console.log("Running auto-installer...");

      // Run the batch script and exit the application
      exec(`"${batchPath}"`, (error) => {
        if (error) {
          console.error("Error running auto-installer:", error);
        }
      });

      // Give the batch script a moment to start, then quit the application
      setTimeout(() => {
        app.quit();
      }, 1000);
    } else {
      // User cancelled, clean up the temporary directory
      try {
        fs.rmSync(tmpDir, { recursive: true, force: true });
        console.log("Cleaned up temporary directory");
      } catch (cleanupError) {
        console.error("Error cleaning up temporary directory:", cleanupError);
      }
    }
  } catch (error) {
    console.error("Error in auto-installer:", error);
    logError(error, "Auto-installer failed");

    await dialog.showMessageBox(mainWindow, {
      type: "error",
      title: "Auto-Installer Error",
      message: `Failed to download or run the auto-installer.\n\nError: ${error.message}\n\nPlease try updating manually.`,
      buttons: ["OK"],
    });
  }
}

// Keep the interval check
setInterval(() => checkForUpdates(true), 4 * 60 * 60 * 1000); // Check every 4 hours

// Update the createWindow function
function createWindow() {
  resetPrinterDataIfNeeded();

  mainWindow = new BrowserWindow({
    width: 800,
    height: 600,
    title: "DHL Printer Manager Alternative",
    autoHideMenuBar: true,
    webPreferences: {
      preload: path.join(__dirname, "preload.js"),
      nodeIntegration: false,
      contextIsolation: true,
    },
  });

  mainWindow.setTitle("DHL Printer Manager Alternative");
  mainWindow.removeMenu();
  mainWindow.loadFile("index.html");

  mainWindow.once("ready-to-show", async () => {
    mainWindow.show();

    // Check if this is the first run and pin to taskbar
    console.log("Portable data directory:", portableDataDir);
    const settings = readSettings();
    console.log("firstRun flag:", settings.firstRun);
    if (settings.firstRun) {
      console.log("First run detected - pinning application to taskbar");
      try {
        await pinMainApplicationToTaskbar();

        // Update settings to mark first run as complete
        const updatedSettings = { ...settings, firstRun: false };
        console.log("About to save updated settings...");
        saveSettings(updatedSettings);
        console.log("First run setup completed");
        console.log("Updated settings:", updatedSettings);

        // Note: Welcome notification removed for cleaner user experience
        console.log("Application pinned to taskbar successfully");
      } catch (error) {
        console.error("Error during first run setup:", error);
        logError(error, "First run taskbar pinning failed");
      }
    } else {
      console.log("Not first run, skipping taskbar pinning");
    }

    // Check for updates after window is ready with a delay
    setTimeout(() => checkForUpdates(true), 2000);
  });

  // Check for updates periodically
  setInterval(() => checkForUpdates(true), UPDATE_CHECK_INTERVAL);
}

// Move these event listeners after the function definition
app.on("ready", createWindow);

app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    app.quit();
  }
});

// Add this at the top with other app event handlers
app.on("before-quit", () => {
  isQuitting = true;
});

// Update the get-printers handler
ipcMain.handle("get-printers", async () => {
  try {
    const printers = await mainWindow.webContents.getPrintersAsync();
    const selectedPrinters = readSelectedPrinters();
    return { printers, selectedPrinters };
  } catch (error) {
    logError(error, "Error fetching printers");
    return { printers: [], selectedPrinters: [] };
  }
});

ipcMain.handle("save-selected-printers", (event, selectedPrinters) => {
  saveSelectedPrinters(selectedPrinters);
});

// Add this utility function at the top of main.js
function delay(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

// Function to pin shortcut to taskbar using pttb.exe
function pinToTaskbar(shortcutPath) {
  return new Promise((resolve, reject) => {
    const pttbPath = path.join(__dirname, "pttb.exe");

    // Check if pttb.exe exists
    if (!fs.existsSync(pttbPath)) {
      const error = new Error(`pttb.exe not found at ${pttbPath}`);
      logError(error, "Pin to taskbar failed");
      reject(error);
      return;
    }

    const command = `"${pttbPath}" "${shortcutPath}"`;

    exec(command, (error, stdout, stderr) => {
      if (error) {
        logError(error, `Error pinning ${shortcutPath} to taskbar`);
        reject(error);
      } else {
        console.log(`Successfully pinned ${shortcutPath} to taskbar`);
        resolve();
      }
    });
  });
}

// Function to unpin shortcut from taskbar using pttb.exe
function unpinFromTaskbar(shortcutPath) {
  return new Promise((resolve, reject) => {
    const pttbPath = path.join(__dirname, "pttb.exe");

    // Check if pttb.exe exists
    if (!fs.existsSync(pttbPath)) {
      const error = new Error(`pttb.exe not found at ${pttbPath}`);
      logError(error, "Unpin from taskbar failed");
      reject(error);
      return;
    }

    const command = `"${pttbPath}" -u "${shortcutPath}"`;

    exec(command, (error, stdout, stderr) => {
      if (error) {
        logError(error, `Error unpinning ${shortcutPath} from taskbar`);
        reject(error);
      } else {
        console.log(`Successfully unpinned ${shortcutPath} from taskbar`);
        resolve();
      }
    });
  });
}

// Function to create main application shortcuts (always both desktop and taskbar)
async function pinMainApplicationToTaskbar() {
  try {
    const desktopPath = app.getPath("desktop");
    const appName = "DHL Printer Manager Alternative";
    const shortcutPath = path.join(desktopPath, `${appName}.lnk`);
    const vbsPath = path.join(desktopPath, `${appName}_app.vbs`);

    // Get the executable path
    const executablePath = process.execPath;
    const iconPath = path.join(__dirname, "Icons", "dhl.ico");

    // Escape paths for VBS
    const escapedExePath = executablePath.replace(/\\/g, "\\\\");
    const escapedIconPath = iconPath.replace(/\\/g, "\\\\");
    const escapedShortcutPath = shortcutPath.replace(/\\/g, "\\\\");

    const vbsContent = `
Set oWS = WScript.CreateObject("WScript.Shell")
sLinkFile = "${escapedShortcutPath}"
Set oLink = oWS.CreateShortcut(sLinkFile)
oLink.TargetPath = "${escapedExePath}"
oLink.WorkingDirectory = "${__dirname.replace(/\\/g, "\\\\")}"
oLink.IconLocation = "${escapedIconPath}"
oLink.Description = "DHL Printer Manager Alternative - Manage printer shortcuts"
oLink.Save
    `.trim();

    // Write and execute VBS
    fs.writeFileSync(vbsPath, vbsContent);

    await new Promise((resolve, reject) => {
      exec(`wscript.exe "${vbsPath}"`, async (error) => {
        await delay(100);
        try {
          if (fs.existsSync(vbsPath)) {
            fs.unlinkSync(vbsPath);
          }
        } catch (deleteError) {
          console.error("Error deleting temp VBS file:", deleteError);
        }

        if (error) {
          reject(error);
        } else {
          resolve();
        }
      });
    });

    // Wait a moment for the shortcut to be created
    await delay(500);

    // Pin the shortcut to taskbar while keeping the desktop shortcut
    if (fs.existsSync(shortcutPath)) {
      await pinToTaskbar(shortcutPath);
      console.log(
        "Successfully created desktop shortcut and pinned DHL Printer Manager Alternative to taskbar"
      );
      // Keep both desktop shortcut and taskbar pin for main application
    } else {
      throw new Error("Shortcut was not created successfully");
    }
  } catch (error) {
    console.error("Error pinning main application to taskbar:", error);
    logError(error, "Failed to pin main application to taskbar");
  }
}

// Function to create a temporary shortcut for taskbar operations
async function createTempShortcut(printerName) {
  const desktopPath = app.getPath("desktop");
  const shortcutPath = path.join(desktopPath, `${printerName}.lnk`);
  const vbsPath = path.join(desktopPath, `${printerName}_temp.vbs`);

  try {
    // Get the full printer name
    const printers = await mainWindow.webContents.getPrintersAsync();
    const fullPrinter = printers.find((p) => p.name.startsWith(printerName));

    if (!fullPrinter) {
      throw new Error(`Printer ${printerName} not found`);
    }

    const iconFile = getIconForPrinter(printerName);
    const escapedIconPath = iconFile.replace(/\\/g, "\\\\");
    const escapedPrinterName = fullPrinter.name.replace(/"/g, '""');

    const vbsContent = `
Set oWS = WScript.CreateObject("WScript.Shell")
sLinkFile = "${shortcutPath.replace(/\\/g, "\\\\")}"
Set oLink = oWS.CreateShortcut(sLinkFile)
oLink.TargetPath = "rundll32.exe"
oLink.Arguments = "printui.dll,PrintUIEntry /y /n ""${escapedPrinterName}"""
oLink.IconLocation = "${escapedIconPath}"
oLink.Save
    `.trim();

    // Write and execute VBS
    fs.writeFileSync(vbsPath, vbsContent);

    return new Promise((resolve, reject) => {
      exec(`wscript.exe "${vbsPath}"`, async (error) => {
        await delay(100);
        try {
          if (fs.existsSync(vbsPath)) {
            fs.unlinkSync(vbsPath);
          }
        } catch (deleteError) {
          console.error("Error deleting temp VBS file:", deleteError);
        }

        if (error) {
          reject(error);
        } else {
          resolve(shortcutPath);
        }
      });
    });
  } catch (error) {
    throw error;
  }
}

// Update create-printer-shortcut handler
ipcMain.handle("create-printer-shortcut", async (event, printerName) => {
  try {
    // Get user settings to determine what to create for printer shortcuts
    const settings = readSettings();
    const shouldCreateDesktop = settings.createDesktopShortcut;
    const shouldCreateTaskbar = settings.createTaskbarShortcut;

    // If neither option is enabled, do nothing
    if (!shouldCreateDesktop && !shouldCreateTaskbar) {
      console.log("No shortcut creation options enabled in settings");
      return true;
    }

    const desktopPath = app.getPath("desktop");
    const shortcutPath = path.join(desktopPath, `${printerName}.lnk`);
    const vbsPath = path.join(desktopPath, `${printerName}_shortcut.vbs`);

    // Get the full printer name
    const printers = await mainWindow.webContents.getPrintersAsync();
    const fullPrinter = printers.find((p) => p.name.startsWith(printerName));

    if (!fullPrinter) {
      throw new Error(`Printer ${printerName} not found`);
    }

    const iconFile = getIconForPrinter(printerName);
    const escapedIconPath = iconFile.replace(/\\/g, "\\\\");
    const escapedPrinterName = fullPrinter.name.replace(/"/g, '""');

    // Create desktop shortcut if enabled
    if (shouldCreateDesktop) {
      const vbsContent = `
Set oWS = WScript.CreateObject("WScript.Shell")
sLinkFile = "${shortcutPath.replace(/\\/g, "\\\\")}"
Set oLink = oWS.CreateShortcut(sLinkFile)
oLink.TargetPath = "rundll32.exe"
oLink.Arguments = "printui.dll,PrintUIEntry /y /n ""${escapedPrinterName}"""
oLink.IconLocation = "${escapedIconPath}"
oLink.Save
      `.trim();

      // Write VBS file with unique name
      fs.writeFileSync(vbsPath, vbsContent);

      // Execute VBS and wait for completion
      return new Promise((resolve) => {
        exec(`wscript.exe "${vbsPath}"`, async (error) => {
          await delay(100); // Wait 100ms before deleting the VBS file
          try {
            if (fs.existsSync(vbsPath)) {
              fs.unlinkSync(vbsPath); // Cleanup VBS
            }
          } catch (deleteError) {
            console.error("Error deleting VBS file:", deleteError);
          }

          if (error) {
            resolve(false);
          } else {
            // If taskbar creation is also enabled and desktop shortcut was successful
            if (shouldCreateTaskbar) {
              // Add a small delay to ensure the shortcut file is fully written
              setTimeout(() => {
                pinToTaskbar(shortcutPath)
                  .then(() => resolve(true))
                  .catch(() => resolve(true)); // Still return true even if taskbar pinning fails
              }, 200);
            } else {
              resolve(true);
            }
          }
        });
      });
    } else if (shouldCreateTaskbar) {
      // Only taskbar pinning requested, but we need a temporary shortcut file
      // Create a temporary shortcut for taskbar pinning, then optionally remove it
      const vbsContent = `
Set oWS = WScript.CreateObject("WScript.Shell")
sLinkFile = "${shortcutPath.replace(/\\/g, "\\\\")}"
Set oLink = oWS.CreateShortcut(sLinkFile)
oLink.TargetPath = "rundll32.exe"
oLink.Arguments = "printui.dll,PrintUIEntry /y /n ""${escapedPrinterName}"""
oLink.IconLocation = "${escapedIconPath}"
oLink.Save
      `.trim();

      // Write VBS file with unique name
      fs.writeFileSync(vbsPath, vbsContent);

      // Execute VBS and wait for completion
      return new Promise((resolve) => {
        exec(`wscript.exe "${vbsPath}"`, async (error) => {
          await delay(100); // Wait 100ms before deleting the VBS file
          try {
            if (fs.existsSync(vbsPath)) {
              fs.unlinkSync(vbsPath); // Cleanup VBS
            }
          } catch (deleteError) {
            console.error("Error deleting VBS file:", deleteError);
          }

          if (error) {
            resolve(false);
          } else {
            // Pin to taskbar, then remove the desktop shortcut
            setTimeout(async () => {
              try {
                await pinToTaskbar(shortcutPath);
                // Remove the temporary desktop shortcut since user only wants taskbar
                if (fs.existsSync(shortcutPath)) {
                  fs.unlinkSync(shortcutPath);
                }
                resolve(true);
              } catch (taskbarError) {
                // If taskbar pinning fails, remove the temporary shortcut
                if (fs.existsSync(shortcutPath)) {
                  fs.unlinkSync(shortcutPath);
                }
                resolve(false);
              }
            }, 200);
          }
        });
      });
    }

    return true;
  } catch (error) {
    logError(error, `Error creating shortcut for ${printerName}`);
    return false;
  }
});

// Update remove-printer-shortcut handler
ipcMain.handle("remove-printer-shortcut", async (event, printerName) => {
  try {
    // Get user settings to determine what to remove for printer shortcuts
    const settings = readSettings();
    const shouldRemoveDesktop = settings.createDesktopShortcut;
    const shouldRemoveTaskbar = settings.createTaskbarShortcut;

    const desktopPath = app.getPath("desktop");
    const shortcutPath = path.join(desktopPath, `${printerName}.lnk`);

    // Handle taskbar removal
    if (shouldRemoveTaskbar) {
      try {
        // For taskbar-only mode, we need to create a temporary shortcut for unpinning
        if (!shouldRemoveDesktop && !fs.existsSync(shortcutPath)) {
          console.log(
            `Creating temporary shortcut for unpinning ${printerName} from taskbar`
          );
          await createTempShortcut(printerName);
        }

        // Now unpin from taskbar
        await unpinFromTaskbar(shortcutPath);
        console.log(`Successfully unpinned ${printerName} from taskbar`);

        // If we created a temporary shortcut just for unpinning, remove it
        if (!shouldRemoveDesktop && fs.existsSync(shortcutPath)) {
          fs.unlinkSync(shortcutPath);
          console.log(`Removed temporary shortcut for ${printerName}`);
        }
      } catch (error) {
        // Log but don't fail the entire operation
        console.warn(
          `Failed to unpin ${printerName} from taskbar:`,
          error.message
        );
        logError(error, `Taskbar unpinning failed for ${printerName}`);
      }
    }

    // Remove desktop shortcut if enabled and it exists
    if (shouldRemoveDesktop && fs.existsSync(shortcutPath)) {
      fs.unlinkSync(shortcutPath);
      console.log(`Removed desktop shortcut for ${printerName}`);
    }

    return true;
  } catch (error) {
    logError(error, `Error removing shortcut for ${printerName}`);
    return false;
  }
});

// Function to set default printer in Windows
async function setDefaultPrinter(printerName) {
  return new Promise((resolve, reject) => {
    console.log(`🖨️ [MAIN] Starting setDefaultPrinter for: ${printerName}`);
    const startTime = Date.now();

    // Use faster rundll32 method instead of slow WMI
    const command = `rundll32 printui.dll,PrintUIEntry /y /n "${printerName}"`;

    console.log(`🔧 [MAIN] Executing rundll32 command...`);
    const commandStartTime = Date.now();

    exec(command, (error, stdout, stderr) => {
      const commandEndTime = Date.now();
      const commandDuration = commandEndTime - commandStartTime;
      const totalDuration = commandEndTime - startTime;

      console.log(`⏱️ [MAIN] rundll32 execution time: ${commandDuration}ms`);
      console.log(`⏱️ [MAIN] Total function time: ${totalDuration}ms`);

      if (error) {
        console.error(
          `❌ [MAIN] Error setting default printer (${totalDuration}ms): ${error.message}`
        );
        console.error(`❌ [MAIN] Command output:`, stdout);
        console.error(`❌ [MAIN] Command error:`, stderr);
        reject(error);
      } else {
        console.log(
          `✅ [MAIN] Successfully set default printer to: ${printerName} (${totalDuration}ms)`
        );
        console.log(`📄 [MAIN] Command output:`, stdout);
        resolve(true);
      }
    });
  });
}

// Function to get current default printer
async function getCurrentDefaultPrinter() {
  return new Promise((resolve, reject) => {
    // Use PowerShell to get the current default printer
    const command = `powershell.exe -Command "& {Get-WmiObject -Class Win32_Printer | Where-Object {$_.Default -eq $true} | Select-Object -ExpandProperty Name}"`;

    exec(command, (error, stdout, stderr) => {
      if (error) {
        console.error(`Error getting default printer: ${error.message}`);
        reject(error);
      } else {
        const defaultPrinter = stdout.trim();
        resolve(defaultPrinter || null);
      }
    });
  });
}

// New IPC handler for setting default printer
ipcMain.handle("set-default-printer", async (event, printerName) => {
  console.log(
    `📨 [IPC] Received set-default-printer request for: ${printerName}`
  );
  const ipcStartTime = Date.now();

  try {
    await setDefaultPrinter(printerName);
    const ipcDuration = Date.now() - ipcStartTime;
    console.log(
      `✅ [IPC] set-default-printer completed successfully (${ipcDuration}ms)`
    );
    return true;
  } catch (error) {
    const ipcDuration = Date.now() - ipcStartTime;
    console.error(
      `❌ [IPC] set-default-printer failed (${ipcDuration}ms):`,
      error
    );
    logError(error, `Error setting default printer to ${printerName}`);
    return false;
  }
});

// New IPC handler for getting current default printer
ipcMain.handle("get-default-printer", async () => {
  try {
    return await getCurrentDefaultPrinter();
  } catch (error) {
    logError(error, "Error getting current default printer");
    return null;
  }
});

ipcMain.handle("save-settings", (event, settings) => {
  saveSettings(settings);
});

ipcMain.handle("get-settings", () => {
  return readSettings();
});

ipcMain.handle("reset-all", () => {
  try {
    // Only delete settings file, keep selected printers
    if (fs.existsSync(settingsFile)) {
      fs.unlinkSync(settingsFile);
    }
    return true;
  } catch (error) {
    console.error("Error resetting settings:", error);
    return false;
  }
});

ipcMain.handle("show-reset-confirmation", async () => {
  const result = await dialog.showMessageBox(mainWindow, {
    type: "question",
    title: "DHL Printer Manager Alternative",
    message: "Are you sure you want to reset all filters and sorting options?",
    buttons: ["Yes", "No"],
    defaultId: 1,
    cancelId: 1,
  });
  return result.response === 0;
});

// Update export-settings handler
ipcMain.handle("export-settings", async () => {
  try {
    const saveDialog = await dialog.showSaveDialog(mainWindow, {
      title: "Export Settings",
      defaultPath: path.join(portableDataDir, "printer-settings.json"),
      filters: [{ name: "JSON Files", extensions: ["json"] }],
    });

    if (!saveDialog.canceled) {
      const exportData = {
        selectedPrinters: readSelectedPrinters(),
        settings: readSettings(),
        exportDate: new Date().toISOString(),
        machineId: getMachineId(),
      };

      fs.writeFileSync(
        saveDialog.filePath,
        JSON.stringify(exportData, null, 2)
      );

      // Show success message with proper title
      await dialog.showMessageBox(mainWindow, {
        type: "info",
        title: "DHL Printer Manager Alternative",
        message: "Settings exported successfully!",
      });

      return true;
    }
    return false;
  } catch (error) {
    logError(error, "Error exporting settings");
    return false;
  }
});

// Update import-settings handler
ipcMain.handle("import-settings", async () => {
  try {
    const result = await dialog.showOpenDialog(mainWindow, {
      title: "Import Settings",
      filters: [{ name: "JSON Files", extensions: ["json"] }],
      properties: ["openFile"],
    });

    if (!result.canceled) {
      const importData = JSON.parse(
        fs.readFileSync(result.filePaths[0], "utf8")
      );

      // Import settings
      saveSettings(importData.settings);
      saveSelectedPrinters(importData.selectedPrinters);

      // Show success message with proper title
      await dialog.showMessageBox(mainWindow, {
        type: "info",
        title: "DHL Printer Manager Alternative",
        message: "Settings imported successfully!",
      });

      return true;
    }
    return false;
  } catch (error) {
    logError(error, "Error importing settings");
    return false;
  }
});

ipcMain.handle("show-message", async (event, message) => {
  await dialog.showMessageBox(mainWindow, {
    type: "info",
    title: "DHL Printer Manager Alternative",
    message: message,
  });
});

ipcMain.handle("show-bulk-create-confirmation", async (event, count) => {
  const result = await dialog.showMessageBox(mainWindow, {
    type: "question",
    title: "DHL Printer Manager Alternative",
    message: `Create shortcuts for ${count} selected printer(s)?`,
    buttons: ["Yes", "No"],
    defaultId: 1,
    cancelId: 1,
  });
  return result.response === 0;
});

ipcMain.handle("show-bulk-remove-confirmation", async (event, count) => {
  const result = await dialog.showMessageBox(mainWindow, {
    type: "question",
    title: "DHL Printer Manager Alternative",
    message: `Remove shortcuts for ${count} selected printer(s)?`,
    buttons: ["Yes", "No"],
    defaultId: 1,
    cancelId: 1,
  });
  return result.response === 0;
});

// Add this logging function
function logError(error, context = "") {
  const timestamp = new Date().toISOString();
  const logMessage = `${timestamp} - ${context}: ${error.stack || error}\n`;

  try {
    fs.appendFileSync(logFile, logMessage);
  } catch (writeError) {
    console.error("Failed to write to log file:", writeError);
  }
}

// Add global error handlers
process.on("uncaughtException", (error) => {
  logError(error, "Uncaught Exception");
});

process.on("unhandledRejection", (error) => {
  logError(error, "Unhandled Rejection");
});

ipcMain.handle("log-error", (event, error, context) => {
  logError(error, context);
});

// Update the IPC handler
ipcMain.handle("check-for-updates", () => checkForUpdates(false));

// Add handler for opening external URLs
ipcMain.handle("open-external", (_, url) => {
  require("electron").shell.openExternal(url);
});

// Add handler for running uninstall
ipcMain.handle("run-complete-cleanup", async () => {
  try {
    const result = await dialog.showMessageBox(mainWindow, {
      type: "warning",
      title: "Uninstall - DHL Printer Manager Alternative",
      message:
        "⚠ WARNING: This will completely remove DHL Printer Manager Alternative!\n\nThis will delete:\n• All application files\n• All printer shortcuts\n• Application data and settings\n• Taskbar pins\n\nThis action cannot be undone!",
      buttons: ["Cancel", "Uninstall"],
      defaultId: 0,
      cancelId: 0,
    });

    if (result.response === 1) {
      // User confirmed, run the cleanup script
      let cleanupScriptPath = path.join(
        __dirname,
        "CleanupDHLPrinterManager-Alternative.ps1"
      );

      // In packaged app, try the main executable directory first
      if (!fs.existsSync(cleanupScriptPath)) {
        const execDir = path.dirname(process.execPath);
        cleanupScriptPath = path.join(
          execDir,
          "CleanupDHLPrinterManager-Alternative.ps1"
        );
      }

      // If still not found, try one level up from resources/app
      if (!fs.existsSync(cleanupScriptPath)) {
        const parentDir = path.dirname(path.dirname(__dirname));
        cleanupScriptPath = path.join(
          parentDir,
          "CleanupDHLPrinterManager-Alternative.ps1"
        );
      }

      console.log("Looking for cleanup script at:", cleanupScriptPath);
      console.log("Script exists:", fs.existsSync(cleanupScriptPath));

      if (!fs.existsSync(cleanupScriptPath)) {
        await dialog.showMessageBox(mainWindow, {
          type: "error",
          title: "Cleanup Script Not Found",
          message: `The cleanup script could not be found.\n\nLooked for: CleanupDHLPrinterManager-Alternative.ps1\nIn directories:\n• ${__dirname}\n• ${path.dirname(
            process.execPath
          )}\n• ${path.dirname(path.dirname(__dirname))}`,
        });
        return false;
      }

      // Show final confirmation
      const finalConfirm = await dialog.showMessageBox(mainWindow, {
        type: "question",
        title: "Final Confirmation",
        message:
          "Are you absolutely sure you want to proceed?\n\nThis will close the application and remove everything.",
        buttons: ["Cancel", "Yes, Remove Everything"],
        defaultId: 0,
        cancelId: 0,
      });

      if (finalConfirm.response === 1) {
        console.log("Starting cleanup process...");
        console.log("Cleanup script path:", cleanupScriptPath);

        // Run the PowerShell cleanup script with better error handling
        const command = `powershell.exe -ExecutionPolicy Bypass -WindowStyle Hidden -File "${cleanupScriptPath}" -NoConfirm`;

        console.log("Executing command:", command);

        exec(command, { timeout: 30000 }, (error, stdout, stderr) => {
          console.log("Cleanup script execution result:");
          console.log("Error:", error);
          console.log("Stdout:", stdout);
          console.log("Stderr:", stderr);

          // Write execution details to a log file for debugging
          const logPath = path.join(
            path.dirname(cleanupScriptPath),
            "cleanup-execution.log"
          );
          const logContent =
            `Cleanup Execution Log - ${new Date().toISOString()}\n` +
            `Command: ${command}\n` +
            `Error: ${error ? error.message : "None"}\n` +
            `Stdout: ${stdout || "None"}\n` +
            `Stderr: ${stderr || "None"}\n` +
            `---\n`;

          try {
            fs.appendFileSync(logPath, logContent);
          } catch (logErr) {
            console.error("Failed to write cleanup log:", logErr);
          }

          if (error) {
            console.error("Error running cleanup script:", error);
            logError(error, "Cleanup script execution failed");
          }
        });

        // Give the script more time to start before quitting
        console.log(
          "Waiting 3 seconds before quitting to ensure script starts..."
        );
        setTimeout(() => {
          console.log("Quitting application...");
          app.quit();
        }, 3000);

        return true;
      }
    }

    return false;
  } catch (error) {
    console.error("Error in uninstall:", error);
    logError(error, "Uninstall failed");

    await dialog.showMessageBox(mainWindow, {
      type: "error",
      title: "Uninstall Error",
      message: `Failed to run uninstall.\n\nError: ${error.message}`,
      buttons: ["OK"],
    });

    return false;
  }
});
