# DHL Printer Manager Alternative - Uninstall Script
# This script removes all files, shortcuts, and data created by DHL Printer Manager Alternative

param(
    [switch]$NoConfirm = $false
)

# Set console title
$Host.UI.RawUI.WindowTitle = "DHL Printer Manager Alternative - Uninstall"

Write-Host "==========================================" -ForegroundColor Red
Write-Host "DHL Printer Manager Alternative - UNINSTALL" -ForegroundColor Red
Write-Host "==========================================" -ForegroundColor Red
Write-Host
Write-Host "⚠ WARNING: This will remove ALL files and shortcuts created by DHL Printer Manager Alternative!" -ForegroundColor Yellow
Write-Host "This includes:" -ForegroundColor Yellow
Write-Host "• Main application files" -ForegroundColor Yellow
Write-Host "• All printer shortcuts on desktop" -ForegroundColor Yellow
Write-Host "• Application data and settings" -ForegroundColor Yellow
Write-Host "• Taskbar pins" -ForegroundColor Yellow
Write-Host

# Confirmation prompt (skip if NoConfirm is specified)
if (-not $NoConfirm) {
    $confirmation = Read-Host "Do you want to proceed? Type 'YES' to confirm"
    if ($confirmation -ne "YES") {
        Write-Host "❌ Cleanup cancelled by user" -ForegroundColor Red
        Write-Host "Press any key to exit..."
        $null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
        exit 0
    }
} else {
    Write-Host "✅ Running in automated mode (no confirmation required)" -ForegroundColor Green
}

Write-Host
Write-Host "🧹 Starting uninstall..." -ForegroundColor Green
Write-Host

# Function to detect OneDrive and determine paths (same as installer)
function Get-InstallationPaths {
    $documentsPath = [Environment]::GetFolderPath("MyDocuments")
    $localAppData = [Environment]::GetFolderPath("LocalApplicationData")
    
    $paths = @()
    
    # Add both possible installation locations
    $paths += @{
        Name = "LocalAppData"
        ZipFile = Join-Path $localAppData "DHLPrinterManager-Alternative.zip"
        ExtractFolder = Join-Path $localAppData "DHLPrinterManager-Alternative"
    }
    
    $paths += @{
        Name = "Documents"
        ZipFile = Join-Path $documentsPath "DHLPrinterManager-Alternative.zip"
        ExtractFolder = Join-Path $documentsPath "DHLPrinterManager-Alternative"
    }
    
    return $paths
}

# Function to close Explorer windows showing a specific folder
function Close-ExplorerWindows {
    param([string]$FolderPath)
    
    if (-not (Test-Path $FolderPath)) {
        return
    }
    
    try {
        Write-Host "  Checking for open Explorer windows..." -ForegroundColor Gray
        
        # Get all Explorer windows
        $shell = New-Object -ComObject Shell.Application
        $windows = $shell.Windows()
        $closedCount = 0
        
        foreach ($window in $windows) {
            try {
                # Check if this is an Explorer window (not Internet Explorer)
                if ($window.Name -eq "Windows Explorer" -or $window.Name -eq "File Explorer") {
                    $windowPath = $window.LocationURL
                    if ($windowPath) {
                        # Convert file:// URL to local path
                        $localPath = [System.Uri]::UnescapeDataString($windowPath) -replace '^file:///', '' -replace '/', '\'
                        
                        # Check if this window is showing our target folder, a subfolder, or has the folder selected
                        if ($localPath -eq $FolderPath -or $localPath.StartsWith($FolderPath + '\') -or $FolderPath.StartsWith($localPath + '\')) {
                            Write-Host "    Closing Explorer window: $localPath" -ForegroundColor Yellow
                            $window.Quit()
                            $closedCount++
                        }
                    }
                }
            }
            catch {
                # Skip windows we can't access
                continue
            }
        }
        
        # Release COM object
        [System.Runtime.Interopservices.Marshal]::ReleaseComObject($shell) | Out-Null
        
        if ($closedCount -gt 0) {
            Write-Host "  [OK] Closed $closedCount Explorer window(s)" -ForegroundColor Green
            # Give Windows a moment to close the windows
            Start-Sleep -Milliseconds 500
        } else {
            Write-Host "  [OK] No Explorer windows found for target folder" -ForegroundColor Green
        }
    }
    catch {
        Write-Host "  ⚠ Could not check for Explorer windows: $($_.Exception.Message)" -ForegroundColor Yellow
        Write-Host "    Continuing anyway..." -ForegroundColor Gray
    }
}

# Function to force remove a file or folder with retries (enhanced version)
function Remove-ItemWithRetry {
    param(
        [Parameter(Mandatory)]
        [string]$Path,
        [int]$MaxRetries = 5,
        [int]$DelayMs = 2000
    )
    
    for ($i = 1; $i -le $MaxRetries; $i++) {
        try {
            if (Test-Path $Path) {
                # On first attempt, close any Explorer windows that might be showing the folder
                if ($i -eq 1) {
                    Close-ExplorerWindows -FolderPath $Path
                }
                
                # On second retry, try to kill any processes that might be locking the files
                if ($i -eq 2) {
                    Write-Host "  Attempting to close any processes using the files..." -ForegroundColor Gray
                    $processNames = @("DHLPrinterManager-Alternative", "printer-app-alternative")
                    foreach ($processName in $processNames) {
                        $processes = Get-Process -Name $processName -ErrorAction SilentlyContinue
                        foreach ($process in $processes) {
                            try {
                                $process.Kill()
                                $process.WaitForExit(2000)
                            } catch { }
                        }
                    }
                }
                
                # Try to remove read-only and hidden attributes first
                if ((Get-Item $Path -Force).PSIsContainer) {
                    # It's a directory
                    Get-ChildItem $Path -Recurse -Force -ErrorAction SilentlyContinue | ForEach-Object {
                        try {
                            $_.Attributes = [System.IO.FileAttributes]::Normal
                        } catch { }
                    }
                    try {
                        (Get-Item $Path -Force).Attributes = [System.IO.FileAttributes]::Normal
                    } catch { }
                } else {
                    # It's a file
                    try {
                        (Get-Item $Path -Force).Attributes = [System.IO.FileAttributes]::Normal
                    } catch { }
                }
                
                Remove-Item $Path -Recurse -Force -ErrorAction Stop
                return $true
            } else {
                return $true  # Already doesn't exist
            }
        }
        catch {
            if ($i -eq $MaxRetries) {
                Write-Host "  [ERROR] Final attempt failed: $($_.Exception.Message)" -ForegroundColor Red
                return $false
            } else {
                Write-Host "  ⚠ Attempt $i failed, waiting $DelayMs ms before retry..." -ForegroundColor Yellow
                Start-Sleep -Milliseconds $DelayMs
            }
        }
    }
    return $false
}

# Function to close any running DHL Printer Manager Alternative processes
function Stop-DHLProcesses {
    Write-Host "1. Stopping DHL Printer Manager Alternative processes..." -ForegroundColor Cyan
    
    $processNames = @("DHLPrinterManager-Alternative", "printer-app-alternative")
    $stoppedAny = $false
    
    foreach ($processName in $processNames) {
        $processes = Get-Process -Name $processName -ErrorAction SilentlyContinue
        if ($processes) {
            $stoppedAny = $true
            foreach ($process in $processes) {
                try {
                    Write-Host "  Stopping process: $($process.ProcessName) (ID: $($process.Id))" -ForegroundColor Gray
                    
                    # Try graceful shutdown first
                    $process.CloseMainWindow()
                    if (-not $process.WaitForExit(3000)) {
                        # Force kill if graceful shutdown fails
                        $process.Kill()
                        $process.WaitForExit(5000)
                    }
                } catch {
                    # If normal methods fail, try taskkill
                    try {
                        Write-Host "  Using taskkill for stubborn process..." -ForegroundColor Gray
                        taskkill /f /pid $process.Id 2>$null
                    } catch {
                        Write-Host "  ⚠ Could not stop process $($process.ProcessName): $($_.Exception.Message)" -ForegroundColor Yellow
                    }
                }
            }
        }
    }
    
    if ($stoppedAny) {
        Write-Host "  Waiting for processes to fully terminate..." -ForegroundColor Gray
        Start-Sleep -Seconds 3
        
        # Double-check that processes are really gone
        foreach ($processName in $processNames) {
            $remainingProcesses = Get-Process -Name $processName -ErrorAction SilentlyContinue
            if ($remainingProcesses) {
                Write-Host "  Force-killing remaining $processName processes..." -ForegroundColor Yellow
                foreach ($process in $remainingProcesses) {
                    try {
                        taskkill /f /pid $process.Id 2>$null
                    } catch { }
                }
            }
        }
        
        Start-Sleep -Seconds 2
        Write-Host "[OK] Processes stopped" -ForegroundColor Green
    } else {
        Write-Host "[OK] No running processes found" -ForegroundColor Green
    }
}

# Function to remove all printer shortcuts from desktop
function Remove-PrinterShortcuts {
    Write-Host "2. Removing printer shortcuts from desktop..." -ForegroundColor Cyan
    
    $desktopPath = [Environment]::GetFolderPath("Desktop")
    $removedCount = 0
    $errorCount = 0
    
    # Get all .lnk files on desktop
    $shortcuts = Get-ChildItem -Path $desktopPath -Filter "*.lnk" -ErrorAction SilentlyContinue
    
    foreach ($shortcut in $shortcuts) {
        try {
            # Read shortcut properties to check if it's a printer shortcut
            $shell = New-Object -ComObject WScript.Shell
            $linkDetails = $shell.CreateShortcut($shortcut.FullName)
            
            # Check if it's a printer shortcut (rundll32.exe with printui.dll)
            if ($linkDetails.TargetPath -like "*rundll32.exe*" -and $linkDetails.Arguments -like "*printui.dll*") {
                Write-Host "  Removing printer shortcut: $($shortcut.Name)" -ForegroundColor Gray
                
                # First try to unpin from taskbar if pttb.exe is available
                $pttbPath = $null
                
                # Try multiple locations for pttb.exe
                $possiblePaths = @(
                    (Join-Path $PSScriptRoot "pttb.exe"),
                    (Join-Path (Get-Location) "pttb.exe"),
                    (Join-Path (Split-Path $PSCommandPath -Parent) "pttb.exe"),
                    (Join-Path (Split-Path (Get-Process -Id $PID).Path -Parent) "pttb.exe")
                )
                
                foreach ($path in $possiblePaths) {
                    if (Test-Path $path) {
                        $pttbPath = $path
                        break
                    }
                }
                
                if ($pttbPath) {
                    try {
                        $unpinCommand = "& `"$pttbPath`" -u `"$($shortcut.FullName)`""
                        Invoke-Expression $unpinCommand | Out-Null
                    } catch {
                        # Ignore unpin errors, continue with shortcut removal
                    }
                }
                
                if (Remove-ItemWithRetry -Path $shortcut.FullName) {
                    $removedCount++
                } else {
                    Write-Host "  ⚠ Failed to remove shortcut: $($shortcut.Name)" -ForegroundColor Yellow
                    $errorCount++
                }
            }
        } catch {
            Write-Host "  ⚠ Error processing $($shortcut.Name): $($_.Exception.Message)" -ForegroundColor Yellow
            $errorCount++
        }
    }
    
    Write-Host "[OK] Removed $removedCount printer shortcuts" -ForegroundColor Green
    if ($errorCount -gt 0) {
        Write-Host "⚠ $errorCount shortcuts had errors during removal" -ForegroundColor Yellow
    }
}

# Function to remove main application shortcuts and taskbar pins
function Remove-MainAppShortcuts {
    Write-Host "3. Removing main application shortcuts..." -ForegroundColor Cyan
    
    $desktopPath = [Environment]::GetFolderPath("Desktop")
    $mainAppShortcut = Join-Path $desktopPath "DHL Printer Manager Alternative.lnk"
    
    # Remove desktop shortcut for main application
    if (Test-Path $mainAppShortcut) {
        # Try to unpin from taskbar first
        $pttbPath = Join-Path $PSScriptRoot "pttb.exe"
        if (Test-Path $pttbPath) {
            try {
                Write-Host "  Unpinning main application from taskbar..." -ForegroundColor Gray
                $unpinCommand = "& `"$pttbPath`" -u `"$mainAppShortcut`""
                Invoke-Expression $unpinCommand | Out-Null
            } catch {
                Write-Host "  ⚠ Could not unpin from taskbar: $($_.Exception.Message)" -ForegroundColor Yellow
            }
        }
        
        try {
            if (Remove-ItemWithRetry -Path $mainAppShortcut) {
                Write-Host "[OK] Removed main application desktop shortcut" -ForegroundColor Green
            } else {
                Write-Host "⚠ Could not remove desktop shortcut" -ForegroundColor Yellow
            }
        }
        catch {
            Write-Host "⚠ Could not remove desktop shortcut: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    } else {
        Write-Host "[OK] No main application desktop shortcut found" -ForegroundColor Green
    }
}

# Function to remove all installation files
function Remove-InstallationFiles {
    Write-Host "4. Removing installation files..." -ForegroundColor Cyan
    
    $paths = Get-InstallationPaths
    $removedAny = $false
    
    foreach ($pathSet in $paths) {
        Write-Host "  Checking $($pathSet.Name) location..." -ForegroundColor Gray
        
        # Remove application folder
        if (Test-Path $pathSet.ExtractFolder) {
            Write-Host "    Found application folder: $($pathSet.ExtractFolder)" -ForegroundColor Gray
            if (Remove-ItemWithRetry -Path $pathSet.ExtractFolder -MaxRetries 5 -DelayMs 2000) {
                Write-Host "    [OK] Removed application folder" -ForegroundColor Green
                $removedAny = $true
            } else {
                Write-Host "    ⚠ Could not remove application folder" -ForegroundColor Yellow
            }
        }
        
        # Remove zip file
        if (Test-Path $pathSet.ZipFile) {
            Write-Host "    Found zip file: $($pathSet.ZipFile)" -ForegroundColor Gray
            if (Remove-ItemWithRetry -Path $pathSet.ZipFile) {
                Write-Host "    [OK] Removed zip file" -ForegroundColor Green
                $removedAny = $true
            } else {
                Write-Host "    ⚠ Could not remove zip file" -ForegroundColor Yellow
            }
        }
    }
    
    if (-not $removedAny) {
        Write-Host "[OK] No installation files found" -ForegroundColor Green
    }
}

# Function to remove application data and settings
function Remove-ApplicationData {
    Write-Host "5. Removing application data and settings..." -ForegroundColor Cyan
    
    # AppData path for Electron app (based on package.json name "printer-app-alternative")
    $appDataPath = Join-Path $env:APPDATA "printer-app-alternative"
    
    if (Test-Path $appDataPath) {
        Write-Host "  Found AppData folder: $appDataPath" -ForegroundColor Gray
        
        # Close any Explorer windows showing this folder first
        Close-ExplorerWindows -FolderPath $appDataPath
        
        # Give a moment for file handles to be released
        Write-Host "  Waiting for file handles to be released..." -ForegroundColor Gray
        Start-Sleep -Milliseconds 2000
        
        # Try to force remove with retries (using same settings as autoinstall)
        if (Remove-ItemWithRetry -Path $appDataPath -MaxRetries 5 -DelayMs 2000) {
            Write-Host "[OK] Removed application data folder" -ForegroundColor Green
        } else {
            Write-Host "⚠ Could not remove AppData folder after multiple attempts" -ForegroundColor Yellow
            Write-Host "  Some files may still be locked by the system." -ForegroundColor Yellow
            Write-Host "  You can manually delete: $appDataPath" -ForegroundColor Yellow
            Write-Host "  Or restart your computer to release all file handles." -ForegroundColor Yellow
        }
    } else {
        Write-Host "[OK] No application data folder found" -ForegroundColor Green
    }
    
    # Also check for portable settings files next to the executable
    $portableDataFiles = @("settings.json", "selectedPrinters.json", "machineId.json")
    $currentDir = Split-Path $PSCommandPath -Parent
    $removedPortableFiles = 0
    
    foreach ($fileName in $portableDataFiles) {
        $filePath = Join-Path $currentDir $fileName
        if (Test-Path $filePath) {
            Write-Host "  Found portable data file: $fileName" -ForegroundColor Gray
            if (Remove-ItemWithRetry -Path $filePath) {
                $removedPortableFiles++
            }
        }
    }
    
    if ($removedPortableFiles -gt 0) {
        Write-Host "[OK] Removed $removedPortableFiles portable data files" -ForegroundColor Green
    }
}

# Function to remove temporary files and logs
function Remove-TemporaryFiles {
    Write-Host "6. Removing temporary files and logs..." -ForegroundColor Cyan
    
    $tempPaths = @(
        (Join-Path $env:TEMP "*DHLPrinterManager-Alternative*"),
        (Join-Path $env:TEMP "*dhl-update*"),
        (Join-Path $env:TEMP "*printer-app-alternative*")
    )
    
    $removedCount = 0
    
    foreach ($tempPattern in $tempPaths) {
        $tempItems = Get-ChildItem -Path $tempPattern -ErrorAction SilentlyContinue
        foreach ($item in $tempItems) {
            try {
                Write-Host "  Removing temp item: $($item.Name)" -ForegroundColor Gray
                if (Remove-ItemWithRetry -Path $item.FullName) {
                    $removedCount++
                }
            } catch {
                Write-Host "  ⚠ Could not remove $($item.Name): $($_.Exception.Message)" -ForegroundColor Yellow
            }
        }
    }
    
    if ($removedCount -gt 0) {
        Write-Host "[OK] Removed $removedCount temporary items" -ForegroundColor Green
    } else {
        Write-Host "[OK] No temporary files found" -ForegroundColor Green
    }
}

# Function to clean registry entries (optional, for thorough cleanup)
function Remove-RegistryEntries {
    Write-Host "7. Cleaning registry entries..." -ForegroundColor Cyan
    
    $registryPaths = @(
        "HKCU:\Software\printer-app-alternative",
        "HKCU:\Software\DHLPrinterManager-Alternative"
    )
    
    $removedCount = 0
    
    foreach ($regPath in $registryPaths) {
        if (Test-Path $regPath) {
            try {
                Write-Host "  Removing registry key: $regPath" -ForegroundColor Gray
                Remove-Item -Path $regPath -Recurse -Force
                $removedCount++
            } catch {
                Write-Host "  Warning: Could not remove registry key" -ForegroundColor Yellow
            }
        }
    }
    
    if ($removedCount -gt 0) {
        Write-Host "[OK] Removed $removedCount registry entries" -ForegroundColor Green
    } else {
        Write-Host "[OK] No registry entries found" -ForegroundColor Green
    }
}

# Main cleanup execution
try {
    Stop-DHLProcesses
    Remove-PrinterShortcuts
    Remove-MainAppShortcuts
    Remove-InstallationFiles
    Remove-ApplicationData
    Remove-TemporaryFiles
    Remove-RegistryEntries
    
    Write-Host
    Write-Host "✅ CLEANUP COMPLETED SUCCESSFULLY!" -ForegroundColor Green
    Write-Host
    Write-Host "All DHL Printer Manager Alternative files and shortcuts have been removed." -ForegroundColor Green
    Write-Host "The application has been completely uninstalled from this system." -ForegroundColor Green
    
} catch {
    Write-Host
    Write-Host "❌ CLEANUP ENCOUNTERED ERRORS" -ForegroundColor Red
    Write-Host "Error: $($_.Exception.Message)" -ForegroundColor Red
    Write-Host "Some files may need to be removed manually." -ForegroundColor Yellow
}

Write-Host
Write-Host "Press any key to exit..."
$null = $Host.UI.RawUI.ReadKey("NoEcho,IncludeKeyDown")
