﻿# DHL Printer Manager Alternative - Auto Download and Install (PowerShell)
# Enhanced version with better SSL/TLS handling

# Set console title
$Host.UI.RawUI.WindowTitle = "DHL Printer Manager Alternative - Auto Download and Install"

Write-Host "==========================================" -ForegroundColor Green
Write-Host "DHL Printer Manager Alternative - Auto Installer" -ForegroundColor Green
Write-Host "==========================================" -ForegroundColor Green
Write-Host

# Configuration - Smart path selection based on OneDrive detection
# Function to detect OneDrive and determine installation path
function Get-InstallationPath {
    $documentsPath = [Environment]::GetFolderPath("MyDocuments")
    $localAppData = [Environment]::GetFolderPath("LocalApplicationData")
    
    if ($documentsPath -like "*OneDrive*" -or (Get-Process -Name "OneDrive" -ErrorAction SilentlyContinue)) {
        Write-Host "⚠ OneDrive detected - Using local installation path to avoid sync conflicts" -ForegroundColor Yellow
        $installPath = Join-Path $localAppData "DHLPrinterManager-Alternative"
        Write-Host "  Installation will use: $installPath" -ForegroundColor Cyan
        return @{
            ZipFile = Join-Path $localAppData "DHLPrinterManager-Alternative.zip"
            ExtractFolder = $installPath
            Location = "LocalApplicationData"
        }
    } else {
        Write-Host "✓ OneDrive not detected - Using Documents folder for installation" -ForegroundColor Green
        $installPath = Join-Path $documentsPath "DHLPrinterManager-Alternative"
        Write-Host "  Installation will use: $installPath" -ForegroundColor Cyan
        return @{
            ZipFile = Join-Path $documentsPath "DHLPrinterManager-Alternative.zip"
            ExtractFolder = $installPath
            Location = "Documents"
        }
    }
}

# Get installation paths based on OneDrive detection
$InstallPaths = Get-InstallationPath
$DOWNLOAD_URL = "https://download.teambunny.co/dhl/Programs/PrinterManager/releases/DHLPrinterManager-Alternative.zip"
$ZIP_FILE = $InstallPaths.ZipFile
$EXTRACT_FOLDER = $InstallPaths.ExtractFolder

# Function to detect OneDrive and warn user (legacy function for compatibility)
function Test-OneDriveSync {
    $documentsPath = [Environment]::GetFolderPath("MyDocuments")
    if ($documentsPath -like "*OneDrive*" -or (Get-Process -Name "OneDrive" -ErrorAction SilentlyContinue)) {
        return $true
    }
    return $false
}

Write-Host "[0/4] Performing cleanup and preparation..." -ForegroundColor Yellow
Write-Host

# Function to close running DHL Printer Manager processes
function Stop-DHLPrinterManager {
    Write-Host "Checking for running DHL Printer Manager processes..." -ForegroundColor Cyan
    
    $processNames = @("DHLPrinterManager", "DHL*", "PrinterManager*")
    $foundProcesses = @()
    
    foreach ($processName in $processNames) {
        $processes = Get-Process -Name $processName -ErrorAction SilentlyContinue
        if ($processes) {
            $foundProcesses += $processes
        }
    }
    
    if ($foundProcesses.Count -gt 0) {
        Write-Host "Found $($foundProcesses.Count) running DHL Printer Manager process(es):" -ForegroundColor Yellow
        foreach ($proc in $foundProcesses) {
            Write-Host "  - $($proc.ProcessName) (PID: $($proc.Id))" -ForegroundColor Gray
        }
        
        Write-Host "Attempting to close processes gracefully..." -ForegroundColor Cyan
        foreach ($proc in $foundProcesses) {
            try {
                $proc.CloseMainWindow()
                if (!$proc.WaitForExit(5000)) {
                    Write-Host "  Forcefully terminating $($proc.ProcessName)..." -ForegroundColor Yellow
                    $proc.Kill()
                }
                Write-Host "  [OK] Closed $($proc.ProcessName)" -ForegroundColor Green
            }
            catch {
                Write-Host "  [ERROR] Failed to close $($proc.ProcessName): $($_.Exception.Message)" -ForegroundColor Red
            }
        }
    } else {
        Write-Host "[OK] No running DHL Printer Manager processes found" -ForegroundColor Green
    }
}

# Function to close Explorer windows showing a specific folder
function Close-ExplorerWindows {
    param([string]$FolderPath)
    
    if (-not (Test-Path $FolderPath)) {
        return
    }
    
    try {
        Write-Host "Checking for open Explorer windows..." -ForegroundColor Cyan
        
        # Get all Explorer windows
        $shell = New-Object -ComObject Shell.Application
        $windows = $shell.Windows()
        $closedCount = 0
        
        foreach ($window in $windows) {
            try {
                # Check if this is an Explorer window (not Internet Explorer)
                if ($window.Name -eq "Windows Explorer" -or $window.Name -eq "File Explorer") {
                    $windowPath = $window.LocationURL
                    if ($windowPath) {
                        # Convert file:// URL to local path
                        $localPath = [System.Uri]::UnescapeDataString($windowPath) -replace '^file:///', '' -replace '/', '\'
                        
                        # Check if this window is showing our target folder, a subfolder, or has the folder selected
                        if ($localPath -eq $FolderPath -or $localPath.StartsWith($FolderPath + '\') -or $FolderPath.StartsWith($localPath + '\')) {
                            Write-Host "  Closing Explorer window: $localPath" -ForegroundColor Yellow
                            $window.Quit()
                            $closedCount++
                        }
                    }
                }
            }
            catch {
                # Skip windows we can't access
                continue
            }
        }
        
        # Release COM object
        [System.Runtime.Interopservices.Marshal]::ReleaseComObject($shell) | Out-Null
        
        if ($closedCount -gt 0) {
            Write-Host "[OK] Closed $closedCount Explorer window(s)" -ForegroundColor Green
            # Give Windows a moment to close the windows
            Start-Sleep -Milliseconds 500
        } else {
            Write-Host "[OK] No Explorer windows found for target folder" -ForegroundColor Green
        }
    }
    catch {
        Write-Host "⚠ Could not check for Explorer windows: $($_.Exception.Message)" -ForegroundColor Yellow
        Write-Host "  Continuing anyway..." -ForegroundColor Gray
    }
}

# Function to force remove a file or folder with retries
function Remove-ItemWithRetry {
    param(
        [Parameter(Mandatory)]
        [string]$Path,
        [int]$MaxRetries = 3,
        [int]$DelayMs = 1000
    )
    
    for ($i = 1; $i -le $MaxRetries; $i++) {
        try {
            if (Test-Path $Path) {
                # Try to remove read-only attributes first
                if ((Get-Item $Path -Force).PSIsContainer) {
                    # It's a directory
                    Get-ChildItem $Path -Recurse -Force | ForEach-Object {
                        try {
                            $_.Attributes = $_.Attributes -band (-bnot [System.IO.FileAttributes]::ReadOnly)
                        } catch { }
                    }
                } else {
                    # It's a file
                    try {
                        (Get-Item $Path -Force).Attributes = (Get-Item $Path -Force).Attributes -band (-bnot [System.IO.FileAttributes]::ReadOnly)
                    } catch { }
                }
                
                Remove-Item $Path -Recurse -Force -ErrorAction Stop
                return $true
            } else {
                return $true  # Already doesn't exist
            }
        }
        catch {
            if ($i -eq $MaxRetries) {
                Write-Host "  [ERROR] Final attempt failed: $($_.Exception.Message)" -ForegroundColor Red
                return $false
            } else {
                Write-Host "  ⚠ Attempt $i failed, waiting $DelayMs ms before retry..." -ForegroundColor Yellow
                Start-Sleep -Milliseconds $DelayMs
            }
        }
    }
    return $false
}

# Function to clean up existing files and folders
function Remove-ExistingFiles {
    Write-Host "Checking for existing files and folders..." -ForegroundColor Cyan
    
    # Check and remove existing ZIP file
    if (Test-Path $ZIP_FILE) {
        Write-Host "Removing existing ZIP file..." -ForegroundColor Yellow
        if (Remove-ItemWithRetry -Path $ZIP_FILE) {
            Write-Host "[OK] Removed existing ZIP file: $ZIP_FILE" -ForegroundColor Green
        } else {
            Write-Host "[ERROR] Failed to remove ZIP file after multiple attempts" -ForegroundColor Red
            Write-Host "  Please manually delete: $ZIP_FILE" -ForegroundColor Yellow
            Read-Host "Press Enter once you have deleted the file to continue"
        }
    } else {
        Write-Host "[OK] No existing ZIP file found" -ForegroundColor Green
    }
    
    # Check and remove existing extraction folder
    if (Test-Path $EXTRACT_FOLDER) {
        Write-Host "Removing existing installation folder..." -ForegroundColor Yellow
        
        # Close any Explorer windows showing this folder first
        Close-ExplorerWindows -FolderPath $EXTRACT_FOLDER
        
        # Give a moment for file handles to be released
        Write-Host "  Waiting for file handles to be released..." -ForegroundColor Gray
        Start-Sleep -Milliseconds 2000
        
        # Try to force remove with retries
        if (Remove-ItemWithRetry -Path $EXTRACT_FOLDER -MaxRetries 5 -DelayMs 2000) {
            Write-Host "[OK] Removed existing installation folder: $EXTRACT_FOLDER" -ForegroundColor Green
        } else {
            Write-Host "[ERROR] Failed to remove installation folder after multiple attempts" -ForegroundColor Red
            Write-Host "  Some files may still be locked by the system." -ForegroundColor Yellow
            Write-Host "  Please manually delete: $EXTRACT_FOLDER" -ForegroundColor Yellow
            Write-Host "  Or restart your computer to release all file handles." -ForegroundColor Yellow
            Read-Host "Press Enter once you have deleted the folder to continue"
        }
    } else {
        Write-Host "[OK] No existing installation folder found" -ForegroundColor Green
    }
}

# Function to remove main application shortcuts (desktop and taskbar)
function Remove-MainAppShortcuts {
    Write-Host "Removing main application shortcuts..." -ForegroundColor Cyan
    
    $desktopPath = [Environment]::GetFolderPath("Desktop")
    $mainAppShortcut = Join-Path $desktopPath "DHL Printer Manager Alternative.lnk"
    
    # Remove desktop shortcut for main application
    if (Test-Path $mainAppShortcut) {
        try {
            Remove-Item $mainAppShortcut -Force
            Write-Host "[OK] Removed main application desktop shortcut" -ForegroundColor Green
        }
        catch {
            Write-Host "⚠ Could not remove desktop shortcut: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    } else {
        Write-Host "[OK] No main application desktop shortcut found" -ForegroundColor Green
    }
    
    # Remove taskbar pin for main application using pttb.exe
    $pttbPath = $null
    
    # Try to find pttb.exe in multiple locations
    $possiblePttbLocations = @(
        (Join-Path $PSScriptRoot "pttb.exe"),
        (Join-Path $EXTRACT_FOLDER "pttb.exe"),
        (Join-Path $EXTRACT_FOLDER "resources\app\pttb.exe")
    )
    
    foreach ($location in $possiblePttbLocations) {
        if (Test-Path $location) {
            $pttbPath = $location
            Write-Host "  Found pttb.exe at: $location" -ForegroundColor Gray
            break
        }
    }
    
    if ($pttbPath) {
        try {
            Write-Host "Unpinning main application from taskbar..." -ForegroundColor Cyan
            
            # Create a temporary shortcut for unpinning if desktop shortcut doesn't exist
            $tempShortcutCreated = $false
            if (-not (Test-Path $mainAppShortcut)) {
                Write-Host "  Creating temporary shortcut for unpinning..." -ForegroundColor Gray
                
                # Find the current DHL Printer Manager executable with broader search
                $currentExe = $null
                if (Test-Path $EXTRACT_FOLDER) {
                    $exeFiles = Get-ChildItem -Path $EXTRACT_FOLDER -Recurse -Filter "*.exe" -ErrorAction SilentlyContinue
                    $currentExe = $exeFiles | Where-Object { 
                        $_.BaseName -like "*DHLPrinterManager-Alternative*" -or 
                        $_.BaseName -eq "DHLPrinterManager-Alternative"
                    } | Select-Object -First 1
                }
                
                if ($currentExe) {
                    # Create a temporary VBS script to create the shortcut
                    $desktopForTemp = [Environment]::GetFolderPath("Desktop")
                    $vbsPath = Join-Path $desktopForTemp "temp_main_app.vbs"
                    
                    # Try to find icon in application folder first, then fallback to script directory
                    $iconPath = $null
                    $possibleIconPaths = @(
                        (Join-Path $currentExe.DirectoryName "Icons\dhl.ico"),
                        (Join-Path $EXTRACT_FOLDER "Icons\dhl.ico"),
                        (Join-Path $EXTRACT_FOLDER "resources\app\Icons\dhl.ico"),
                        (Join-Path $PSScriptRoot "Icons\dhl.ico")
                    )
                    
                    foreach ($iconLoc in $possibleIconPaths) {
                        if (Test-Path $iconLoc) {
                            $iconPath = $iconLoc
                            break
                        }
                    }
                    
                    if (-not $iconPath) {
                        $iconPath = Join-Path $PSScriptRoot "Icons\dhl.ico"  # Fallback
                    }
                    
                    $vbsContent = @"
Set oWS = WScript.CreateObject("WScript.Shell")
sLinkFile = "$($mainAppShortcut.Replace('\', '\\'))"
Set oLink = oWS.CreateShortcut(sLinkFile)
oLink.TargetPath = "$($currentExe.FullName.Replace('\', '\\'))"
oLink.WorkingDirectory = "$($currentExe.DirectoryName.Replace('\', '\\'))"
oLink.IconLocation = "$($iconPath.Replace('\', '\\'))"
oLink.Description = "DHL Printer Manager Alternative - Manage default printer settings"
oLink.Save
"@
                    
                    Set-Content -Path $vbsPath -Value $vbsContent
                    Start-Process -FilePath "wscript.exe" -ArgumentList "`"$vbsPath`"" -Wait -WindowStyle Hidden
                    Start-Sleep -Milliseconds 500
                    
                    # Clean up VBS file
                    if (Test-Path $vbsPath) {
                        Remove-Item $vbsPath -Force -ErrorAction SilentlyContinue
                    }
                    
                    $tempShortcutCreated = $true
                }
            }
            
            # Unpin from taskbar if shortcut exists
            if (Test-Path $mainAppShortcut) {
                Write-Host "  Attempting to unpin from taskbar..." -ForegroundColor Cyan
                
                # Add retry logic for VDI environments
                $maxRetries = 3
                for ($retry = 1; $retry -le $maxRetries; $retry++) {
                    try {
                        if ($retry -gt 1) {
                            Write-Host "    Retry attempt $retry/$maxRetries..." -ForegroundColor Gray
                            Start-Sleep -Seconds 2
                        }
                        
                        $unpinProcess = Start-Process -FilePath $pttbPath -ArgumentList "-u `"$mainAppShortcut`"" -PassThru -Wait -WindowStyle Hidden
                        if ($unpinProcess.ExitCode -eq 0) {
                            Write-Host "[OK] Unpinned main application from taskbar" -ForegroundColor Green
                            break
                        } else {
                            if ($retry -eq $maxRetries) {
                                Write-Host "⚠ Taskbar unpinning failed after $maxRetries attempts (exit code: $($unpinProcess.ExitCode))" -ForegroundColor Yellow
                            }
                        }
                    } catch {
                        if ($retry -eq $maxRetries) {
                            Write-Host "⚠ Taskbar unpinning failed: $($_.Exception.Message)" -ForegroundColor Yellow
                        }
                    }
                }
                
                # Remove the temporary shortcut if we created it
                if ($tempShortcutCreated -and (Test-Path $mainAppShortcut)) {
                    Remove-Item $mainAppShortcut -Force -ErrorAction SilentlyContinue
                    Write-Host "  Cleaned up temporary shortcut" -ForegroundColor Gray
                }
            } else {
                Write-Host "  No shortcut available for unpinning" -ForegroundColor Gray
                
                # Fallback: Try to unpin using common executable patterns
                Write-Host "  Attempting fallback unpinning methods..." -ForegroundColor Cyan
                $commonExePatterns = @(
                    "*DHLPrinterManager-Alternative*.exe",
                    "DHLPrinterManager-Alternative.exe"
                )
                
                foreach ($pattern in $commonExePatterns) {
                    $foundExes = Get-ChildItem -Path $EXTRACT_FOLDER -Recurse -Filter $pattern -ErrorAction SilentlyContinue
                    foreach ($exe in $foundExes) {
                        try {
                            Write-Host "    Trying to unpin: $($exe.BaseName)" -ForegroundColor Gray
                            $fallbackProcess = Start-Process -FilePath $pttbPath -ArgumentList "-u `"$($exe.FullName)`"" -PassThru -Wait -WindowStyle Hidden
                            if ($fallbackProcess.ExitCode -eq 0) {
                                Write-Host "[OK] Successfully unpinned using fallback method" -ForegroundColor Green
                                break
                            }
                        }
                        catch {
                            continue
                        }
                    }
                }
            }
        }
        catch {
            Write-Host "⚠ Could not unpin from taskbar: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    } else {
        Write-Host "⚠ pttb.exe not found, skipping taskbar cleanup" -ForegroundColor Yellow
        Write-Host "  Expected location: $pttbPath" -ForegroundColor Gray
    }
}

# Function to remove AppData folder (user settings and cache)
function Remove-AppDataFolder {
    Write-Host "Removing application data folder..." -ForegroundColor Cyan
    
    # AppData path for Electron app (based on package.json name "printer-app-alternative")
    $appDataPath = Join-Path $env:APPDATA "printer-app-alternative"
    
    if (Test-Path $appDataPath) {
        try {
            Write-Host "  Found AppData folder: $appDataPath" -ForegroundColor Gray
            
            # Close any Explorer windows showing this folder first
            Close-ExplorerWindows -FolderPath $appDataPath
            
            # Give a moment for file handles to be released
            Write-Host "  Waiting for file handles to be released..." -ForegroundColor Gray
            Start-Sleep -Milliseconds 1000
            
            # Try to force remove with retries
            if (Remove-ItemWithRetry -Path $appDataPath -MaxRetries 3 -DelayMs 1000) {
                Write-Host "[OK] Removed application data folder: $appDataPath" -ForegroundColor Green
            } else {
                Write-Host "⚠ Could not remove AppData folder after multiple attempts" -ForegroundColor Yellow
                Write-Host "  This may cause issues with settings from previous versions" -ForegroundColor Yellow
                Write-Host "  You can manually delete: $appDataPath" -ForegroundColor Yellow
            }
        }
        catch {
            Write-Host "⚠ Error removing AppData folder: $($_.Exception.Message)" -ForegroundColor Yellow
        }
    } else {
        Write-Host "[OK] No application data folder found" -ForegroundColor Green
    }
}

# Perform cleanup
Stop-DHLPrinterManager
Remove-MainAppShortcuts  # Move this BEFORE removing files so we can still find the exe
Remove-ExistingFiles
Remove-AppDataFolder

Write-Host "[OK] Cleanup completed successfully!" -ForegroundColor Green
Write-Host

Write-Host "[1/4] Setting up download tools and downloading DHL Printer Manager..." -ForegroundColor Yellow
Write-Host "From: $DOWNLOAD_URL"
Write-Host "To: $ZIP_FILE"
Write-Host

# Function to install wget if not available
function Install-Wget {
    Write-Host "Checking for wget..." -ForegroundColor Cyan
    
    # Check if wget is already available and actually works
    try {
        $null = & wget.exe --version 2>&1
        if ($LASTEXITCODE -eq 0) {
            Write-Host "[OK] wget is already available and working" -ForegroundColor Green
            return $true
        }
    }
    catch {
        # wget command failed
    }
    
    # Check if wget exists as a file
    $wgetCommand = Get-Command wget -ErrorAction SilentlyContinue
    if ($wgetCommand -and $wgetCommand.Source -and (Test-Path $wgetCommand.Source)) {
        try {
            $wgetTest = & $wgetCommand.Source --version 2>&1
            if ($LASTEXITCODE -eq 0) {
                Write-Host "[OK] wget found at $($wgetCommand.Source)" -ForegroundColor Green
                return $wgetCommand.Source
            }
        }
        catch {
            # wget file exists but doesn't work
        }
    }
    
    Write-Host "wget not found or not working. Installing wget..." -ForegroundColor Yellow
    
    try {
        # Try to install wget via winget (Windows Package Manager)
        if (Get-Command winget -ErrorAction SilentlyContinue) {
            Write-Host "Installing wget via winget..." -ForegroundColor Cyan
            $null = & winget install wget.wget --silent --accept-package-agreements --accept-source-agreements 2>&1
            
            # Refresh PATH
            $env:PATH = [System.Environment]::GetEnvironmentVariable("PATH", "Machine") + ";" + [System.Environment]::GetEnvironmentVariable("PATH", "User")
            
            # Test if wget works now
            try {
                $null = & wget.exe --version 2>&1
                if ($LASTEXITCODE -eq 0) {
                    Write-Host "[OK] wget installed successfully via winget" -ForegroundColor Green
                    return $true
                }
            }
            catch { }
        }
        
        # Fallback: Download wget manually
        Write-Host "Downloading wget manually..." -ForegroundColor Cyan
        
        $wgetUrl = "https://eternallybored.org/misc/wget/1.21.4/64/wget.exe"
        $wgetPath = Join-Path $env:TEMP "wget.exe"
        
        # Use WebClient to download wget (ironically)
        [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]::Tls12
        [System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }
        
        $webClient = New-Object System.Net.WebClient
        $webClient.Headers.Add("User-Agent", "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36")
        $webClient.DownloadFile($wgetUrl, $wgetPath)
        $webClient.Dispose()
        
        if (Test-Path $wgetPath) {
            # Test if downloaded wget works
            try {
                $wgetTest = & $wgetPath --version 2>&1
                if ($LASTEXITCODE -eq 0) {
                    Write-Host "[OK] wget downloaded successfully to $wgetPath" -ForegroundColor Green
                    return $wgetPath
                }
            }
            catch {
                Write-Host "[ERROR] Downloaded wget doesn't work" -ForegroundColor Red
            }
        }
        
    }
    catch {
        Write-Host "[ERROR] Failed to install wget: $($_.Exception.Message)" -ForegroundColor Red
    }
    
    return $false
}

# Function to download using wget
function Start-WgetDownload {
    param(
        [string]$Url,
        [string]$OutputPath,
        [string]$WgetPath = "wget"
    )
    
    try {
        Write-Host "Downloading with wget (SSL bypass enabled)..." -ForegroundColor Cyan
        Write-Host "Using wget at: $WgetPath" -ForegroundColor Gray
        
        $wgetArgs = @(
            "--no-check-certificate",      # Ignore SSL certificate errors
            "--timeout=30",                # 30 second timeout
            "--tries=3",                   # Retry 3 times
            "--continue",                  # Continue partial downloads
            "--progress=bar:force",        # Show progress
            "--user-agent=`"Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36`"",
            "--output-document=`"$OutputPath`"",  # Properly quote the path
            $Url
        )
        
        Write-Host "Running: $WgetPath $($wgetArgs -join ' ')" -ForegroundColor Gray
        
        # Test if wget path exists first
        if ($WgetPath -ne "wget" -and -not (Test-Path $WgetPath)) {
            Write-Host "[ERROR] wget executable not found at: $WgetPath" -ForegroundColor Red
            return $false
        }
        
        $process = Start-Process -FilePath $WgetPath -ArgumentList $wgetArgs -Wait -PassThru -NoNewWindow -RedirectStandardOutput (Join-Path $env:TEMP "wget_output.txt") -RedirectStandardError (Join-Path $env:TEMP "wget_error.txt")
        
        # Show output
        $outputFile = Join-Path $env:TEMP "wget_output.txt"
        $errorFile = Join-Path $env:TEMP "wget_error.txt"
        
        if (Test-Path $outputFile) {
            $output = Get-Content $outputFile -Raw
            if ($output) { Write-Host $output -ForegroundColor Gray }
            Remove-Item $outputFile -Force -ErrorAction SilentlyContinue
        }
        
        if (Test-Path $errorFile) {
            $errorContent = Get-Content $errorFile -Raw
            if ($errorContent) { Write-Host $errorContent -ForegroundColor Red }
            Remove-Item $errorFile -Force -ErrorAction SilentlyContinue
        }
        
        if ((Test-Path $OutputPath) -and (Get-Item $OutputPath).Length -gt 0) {
            Write-Host "[OK] Download successful!" -ForegroundColor Green
            return $true
        } elseif ($process.ExitCode -eq 0) {
            Write-Host "[OK] Download successful!" -ForegroundColor Green
            return $true
        } else {
            Write-Host "[ERROR] wget failed with exit code: $($process.ExitCode)" -ForegroundColor Red
            return $false
        }
    }
    catch {
        Write-Host "[ERROR] wget download failed: $($_.Exception.Message)" -ForegroundColor Red
        return $false
    }
}

# Install wget and attempt download
$wgetResult = Install-Wget

if ($wgetResult -eq $true) {
    # wget is available in PATH
    $downloadResult = Start-WgetDownload -Url $DOWNLOAD_URL -OutputPath $ZIP_FILE
} elseif ($wgetResult -and (Test-Path $wgetResult)) {
    # wget was downloaded to a specific path
    $downloadResult = Start-WgetDownload -Url $DOWNLOAD_URL -OutputPath $ZIP_FILE -WgetPath $wgetResult
} else {
    Write-Host "[ERROR] Could not install or find wget" -ForegroundColor Red
    $downloadResult = $false
}

# Check if download succeeded
if (-not $downloadResult) {
    Write-Host
    Write-Host "============================================" -ForegroundColor Red
    Write-Host "WGET DOWNLOAD FAILED" -ForegroundColor Red
    Write-Host "============================================" -ForegroundColor Red
    Write-Host "Even wget couldn't handle the SSL certificate issues." -ForegroundColor Yellow
    Write-Host
    Write-Host "Opening browser as final fallback..." -ForegroundColor Cyan
    
    try {
        Start-Process $DOWNLOAD_URL
        Write-Host "[OK] Opened download URL in your default browser" -ForegroundColor Green
    }
    catch {
        Write-Host "[ERROR] Could not open browser automatically" -ForegroundColor Red
    }
    
    Write-Host "Please manually download the file from your browser and save it as:" -ForegroundColor Yellow
    Write-Host $ZIP_FILE -ForegroundColor White
    Write-Host
    
    # Wait for user to complete browser download
    Read-Host "Press Enter once you have downloaded the file"
    
    # Check if user downloaded via browser
    if (Test-Path $ZIP_FILE) {
        Write-Host "[OK] Perfect! File downloaded successfully." -ForegroundColor Green
        $downloadResult = $true
    } else {
        Write-Host "File still not found. Please ensure you downloaded it to:" -ForegroundColor Red
        Write-Host $ZIP_FILE -ForegroundColor White
        Read-Host "Press Enter to exit"
        exit 1
    }
}

# Verify download
if (-not (Test-Path $ZIP_FILE)) {
    Write-Host "ERROR: Download failed! The file was not downloaded." -ForegroundColor Red
    Write-Host "Please check your internet connection and try again." -ForegroundColor Yellow
    Read-Host "Press Enter to exit"
    exit 1
}

Write-Host
Write-Host "[2/4] Download completed successfully!" -ForegroundColor Green
$fileSize = (Get-Item $ZIP_FILE).Length
Write-Host "File size: $([math]::Round($fileSize / 1MB, 2)) MB ($fileSize bytes)"
Write-Host

Write-Host "[3/4] Extracting files to $($InstallPaths.Location)..." -ForegroundColor Yellow
Write-Host "Extracting to: $EXTRACT_FOLDER"
Write-Host

# Create extraction folder if it doesn't exist
if (-not (Test-Path $EXTRACT_FOLDER)) {
    New-Item -ItemType Directory -Path $EXTRACT_FOLDER -Force | Out-Null
}

# Extract using PowerShell
try {
    Expand-Archive -Path $ZIP_FILE -DestinationPath $EXTRACT_FOLDER -Force
    Write-Host "[OK] Extraction completed successfully!" -ForegroundColor Green
}
catch {
    Write-Host "[ERROR] Extraction failed: $($_.Exception.Message)" -ForegroundColor Red
    Read-Host "Press Enter to exit"
    exit 1
}

Write-Host
Write-Host "[4/4] Cleaning up and launching program..." -ForegroundColor Yellow

# Delete the zip file
if (Test-Path $ZIP_FILE) {
    Remove-Item $ZIP_FILE -Force
    Write-Host "[OK] Zip file deleted: $ZIP_FILE" -ForegroundColor Green
}

# Find and run the executable
$exeFound = $null

# Look for DHLPrinterManager-Alternative.exe specifically
$exeFiles = Get-ChildItem -Path $EXTRACT_FOLDER -Recurse -Filter "*.exe"
$targetExe = $exeFiles | Where-Object { $_.BaseName -eq "DHLPrinterManager-Alternative" } | Select-Object -First 1

if ($targetExe) {
    $exeFound = $targetExe.FullName
} elseif ($exeFiles.Count -gt 0) {
    # If DHLPrinterManager-Alternative.exe not found, use the first .exe file
    $exeFound = $exeFiles[0].FullName
}

if ($exeFound) {
    Write-Host "[OK] Found executable: $exeFound" -ForegroundColor Green
    Write-Host "Launching DHL Printer Manager Alternative..." -ForegroundColor Cyan
    Write-Host
    
    try {
        Start-Process -FilePath $exeFound
        Write-Host "[OK] Program launched successfully!" -ForegroundColor Green
    }
    catch {
        Write-Host "[ERROR] Failed to launch program: $($_.Exception.Message)" -ForegroundColor Red
        Write-Host "You can manually run: $exeFound" -ForegroundColor Yellow
    }
} else {
    Write-Host "⚠ WARNING: No executable file found in the extracted folder." -ForegroundColor Yellow
    Write-Host "Please check the folder manually: $EXTRACT_FOLDER" -ForegroundColor Cyan
    Write-Host "Opening folder for you..." -ForegroundColor Cyan
    
    try {
        Start-Process -FilePath $EXTRACT_FOLDER
    }
    catch {
        Write-Host "[ERROR] Could not open folder automatically" -ForegroundColor Red
    }
}

Write-Host
Write-Host "==========================================" -ForegroundColor Green
Write-Host "Installation completed!" -ForegroundColor Green
Write-Host "==========================================" -ForegroundColor Green
Write-Host "Program location: $EXTRACT_FOLDER" -ForegroundColor Cyan
Write-Host

