# GitHub Copilot Instructions - DHL Printer Manager

## 📋 Project Overview

This is an Electron-based application for managing printer shortcuts on Windows systems. The application handles OneDrive environments, taskbar pinning, and provides automated installation/cleanup functionality.

## 🎯 Core Principles for AI Assistance

### 1. **ALWAYS Read Before Writing**

- **NEVER** assume file contents - always use `read_file` to check current state
- **VERIFY** existing code patterns before making changes
- **CHECK** for recent manual edits that might affect your changes
- **UNDERSTAND** the full context before proposing solutions

### 2. **Incremental Changes Only**

- Make **ONE** logical change at a time
- Test each change before proceeding to the next
- Avoid large, sweeping refactors without explicit user request
- Preserve existing functionality unless specifically asked to change it

### 3. **File Path Precision**

- **ALWAYS** use absolute paths with proper escaping for Windows
- Account for spaces in paths (use quotes)
- Verify file existence before operations
- Handle both development and packaged app scenarios

### 4. **Windows Environment Awareness**

- Remember this is a **Windows-only** application
- Handle OneDrive desktop redirection properly
- Account for corporate environments and Group Policy restrictions
- Use PowerShell for system operations, not bash

## 🚫 Common Pitfalls to Avoid

### Code Management

- ❌ **DON'T** write code blocks in chat unless specifically requested
- ❌ **DON'T** assume function signatures without checking
- ❌ **DON'T** modify multiple files simultaneously without coordination
- ❌ **DON'T** remove existing functionality without explicit permission
- ✅ **DO** use appropriate edit tools (`replace_string_in_file`, `edit_notebook_file`)
- ✅ **DO** include sufficient context (3-5 lines) in replacements
- ✅ **DO** verify changes with `get_errors` after modifications

### File Operations

- ❌ **DON'T** assume file structure without checking
- ❌ **DON'T** use relative paths in production code
- ❌ **DON'T** ignore file encoding issues (especially PowerShell)
- ✅ **DO** use `list_dir` and `file_search` to understand structure
- ✅ **DO** handle file existence checks properly
- ✅ **DO** use proper error handling for file operations

### Dependencies & Environment

- ❌ **DON'T** suggest installing packages without checking existing dependencies
- ❌ **DON'T** assume cross-platform compatibility
- ❌ **DON'T** ignore Electron-specific patterns and limitations
- ✅ **DO** respect the existing package.json configuration
- ✅ **DO** maintain Electron security best practices
- ✅ **DO** consider portable app requirements

## 🔧 Project-Specific Guidelines

### Electron Architecture

```javascript
// Main Process (main.js)
- Handles IPC communication
- Manages application lifecycle
- Contains business logic for shortcuts and system operations

// Renderer Process (index.html + inline JS)
- UI interactions and display
- Communicates with main process via IPC
- Handles user input and validation

// Preload (preload.js)
- Exposes safe APIs to renderer
- Bridge between main and renderer processes
```

### Critical Components

1. **Taskbar Pinning**: Uses `pttb.exe` with OneDrive-aware path handling
2. **Shortcut Management**: VBScript generation for .lnk file creation
3. **Settings Persistence**: JSON files in userData directory
4. **Update System**: Auto-installer with PowerShell scripts
5. **Cleanup System**: Complete removal via PowerShell script

### OneDrive Handling

```javascript
// Always check OneDrive status before taskbar operations
const oneDriveRunning = await isOneDriveRunning();
const taskbarDesktopPath = await getTaskbarDesktopPath();

// Use local desktop for pttb.exe operations when OneDrive is active
// Copy shortcuts temporarily for taskbar pinning/unpinning
```

### IPC Pattern

```javascript
// Main Process
ipcMain.handle("action-name", async (event, ...args) => {
  // Implementation
});

// Preload
contextBridge.exposeInMainWorld("electronAPI", {
  actionName: (...args) => ipcRenderer.invoke("action-name", ...args),
});

// Renderer
const result = await window.electronAPI.actionName(...args);
```

## 🐛 Debugging Best Practices

### Error Handling

- **ALWAYS** implement proper try-catch blocks
- **LOG** errors with context using the existing `logError` function
- **PROVIDE** meaningful error messages to users
- **HANDLE** edge cases gracefully

### Testing Strategy

- **BUILD** and test after significant changes
- **VERIFY** file operations with both OneDrive active and inactive
- **TEST** in both development and packaged environments
- **CHECK** for memory leaks in long-running operations

### Logging & Diagnostics

```javascript
// Use existing debug logging for shortcut operations
logShortcutDebug("info", "operation_name", {
  context: "relevant data",
  // ... other debug info
});

// Use error logging for exceptions
logError(error, "Context description");
```

## 📝 Code Style Guidelines

### JavaScript/Node.js

- Use `const` and `let` appropriately
- Implement async/await pattern consistently
- Handle promises properly with error catching
- Use descriptive variable and function names

### PowerShell

- Use proper parameter validation
- Implement retry logic for file operations
- Handle permissions and access issues gracefully
- Provide clear progress feedback to users

### HTML/CSS

- Maintain existing theme system (light/dark)
- Keep responsive design principles
- Use semantic HTML elements
- Follow existing naming conventions

## 🚀 Build & Deployment

### Development Workflow

1. Make incremental changes
2. Test in development environment
3. Build portable version: `npm run package-portable`
4. Copy required files: `pttb.exe`, `Autoinstall.ps1`, `CleanupDHLPrinterManager.ps1`
5. Create zip with files in root (no subfolder)
6. Test extraction and functionality

### Version Management

- Update version in `package.json` when making releases
- Maintain backward compatibility with settings files
- Document breaking changes clearly
- Test upgrade scenarios from previous versions

## 🔒 Security Considerations

### Electron Security

- Never disable security features without explicit need
- Validate all user inputs properly
- Use contextIsolation and disable nodeIntegration
- Sanitize file paths and prevent directory traversal

### PowerShell Execution

- Use `-ExecutionPolicy Bypass` only when necessary
- Validate script paths before execution
- Handle script failures gracefully
- Provide user feedback for security prompts

## 📚 Additional Resources

### Key Files to Understand

- `main.js` - Core application logic
- `preload.js` - Secure IPC bridge
- `index.html` - User interface
- `package.json` - Project configuration
- `Autoinstall.ps1` - Installation script
- `CleanupDHLPrinterManager.ps1` - Cleanup script

### Testing Scenarios

- Test with OneDrive active/inactive
- Test in corporate environments
- Test installation/cleanup on fresh systems
- Test upgrade scenarios
- Test error conditions and recovery

---

## ⚡ Quick Checklist for AI Assistants

Before making any changes:

- [ ] Read relevant files to understand current implementation
- [ ] Check for recent manual edits
- [ ] Understand the full context of the requested change
- [ ] Plan incremental, testable changes
- [ ] Consider Windows-specific requirements
- [ ] Account for OneDrive environment scenarios
- [ ] Verify file paths and operations
- [ ] Test changes after implementation
- [ ] Document any breaking changes or new requirements

Remember: **Quality over speed** - it's better to make careful, well-tested changes than to rush and introduce bugs.
