--
-- Advanced Referee 0.1
-- by nano <nano@fooo.org>
--
-- With this module it's possible to give clients referee rights without
-- sharing the referee password. Clients, whose guid has been added to the
-- guidfile, can log in as a referee with the command /reflogin. If the first
-- argument is "silent", no annoucement will be displayed to the players on
-- the server ("xyz is now a referee").
-- Additionally, clients may get referee status automatically when they connect
-- to the server (without an annoucement).
--
-- cvars you might set:
-- - ref_autoref        shall referees automatically become ref when they
--                      connect? (1 or 0, default 1)
-- - ref_guidfile       the file where the guids are saved
--                      (default: refguids.dat)
-- - ref_disabledcmds   a space-separated list of commands refs are not allowed
--                      to perform (default: "")
--
-- console/rcon commands:
-- - addguid <clientnum> [comment]
--   Use this command to add a client to the guidfile. clientnum is the clients'
--   slot id listed in /players and comment is an optional comment you can set
--   to find a specific client in the file easier, e.g. if you want to remove
--   it later. The default comment is the client's current name.
--

moduleName    = "Advanced Referee"
moduleVersion = "0.1"

et.CS_PLAYERS = 689

refGuids      = {}
disabledCmds  = {}

conf = {
  ['guidfile']     = "refguids.dat",
  ['disabledcmds'] = "",
  ['autoref']      = 1
}

function et_InitGame(levelTime, randomSeed, restart)
  et.RegisterModname(string.format("%s %s", moduleName, moduleVersion))
  initCvars()
  initDisabledCmds()
  loadRefGuids()
end

function et_ClientBegin(clientNum)
  if tonumber(conf['autoref']) == 1 then
    local guid = string.lower(et.GetPlayerInfo(clientNum, "cl_guid"))
    if refGuids[guid] then
      et.makeReferee(clientNum)
    end
  end
end

function et_ConsoleCommand()
  if string.lower(et.trap_Argv(0)) == "addguid" then
    return addguid_cmd()
  end
  return 0
end

function et_ClientCommand(clientNum, command)
  local cmd = string.lower(command)

  if cmd == "ref" then
    if disabledCmds[string.lower(et.trap_Argv(1))] then
      local msg = "This command has been disabled."
      et.trap_SendServerCommand(clientNum, string.format("cpm \"%s\"", msg))
      return 1
    end
  end

  if cmd == "reflogin" then
    return reflogin_cmd(clientNum)
  end

  return 0
end

---

function et.isActive(clientNum)
  return et.gentity_get(clientNum, "inuse")
end

function et.isRef(clientNum)
  if tonumber(et.GetPlayerCS(clientNum, "ref")) == 1 then
    return true
  else
    return false
  end
end

function et.makeReferee(clientNum)
  et.gentity_set(clientNum, "sess.referee", 1)

  local cs = et.trap_GetConfigstring(et.CS_PLAYERS + clientNum)
  local newcs = et.Info_SetValueForKey(cs, "ref", 1)
  et.trap_SetConfigstring(et.CS_PLAYERS + clientNum, newcs)
end

function et.GetPlayerInfo(clientNum, key)
  local info = et.trap_GetUserinfo(clientNum)
  return et.Info_ValueForKey(info, key)
end

function et.GetPlayerCS(clientNum, key)
  local cs = et.trap_GetConfigstring(et.CS_PLAYERS + clientNum)
  return et.Info_ValueForKey(cs, key)
end

---

function initCvars()
  for k, v in pairs(conf) do
    local cvar = et.trap_Cvar_Get(string.format("ref_%s", k))
    if cvar ~= "" then
      conf[k] = cvar
    end
  end
end

function initDisabledCmds()
  for bit in string.gfind(conf['disabledcmds'], "([^%s]+)%s*") do
    disabledCmds[string.lower(bit)] = true
  end
end

function loadRefGuids()
  local fd, len = et.trap_FS_FOpenFile(conf['guidfile'], et.FS_READ)

  if len ~= 1 then
    local str = et.trap_FS_Read(fd, len)
    for line in string.gfind(str, "(%x+)") do
      refGuids[string.lower(line)] = true
    end
  end
end

function reflogin_cmd(clientNum)
  local guid = string.lower(et.GetPlayerInfo(clientNum, "cl_guid"))

  if not refGuids[guid] then
    return 0
  end

  if not et.isRef(clientNum) then
    et.makeReferee(clientNum)

    if string.lower(et.trap_Argv(1)) ~= "silent" then
      local name = et.GetPlayerInfo(clientNum, "name")
      local msg = string.format("%s\n^3is now a referee", name)
      et.trap_SendServerCommand(-1, string.format("cp \"%s\"", msg))
    end
  end

  return 1
end

function addguid_cmd()
  if et.trap_Argc() < 2 then
    et.G_Print("Usage: addguid <clientnum> [comment]\n")
    return 1
  end

  local client = tonumber(et.trap_Argv(1))

  if not et.isActive(client) then
    et.G_Print(string.format("%i is not a valid client\n", client))
    return 1
  end

  local comment = et.Q_CleanStr(et.GetPlayerInfo(client, "name"))
  local guid = string.lower(et.GetPlayerInfo(client, "cl_guid"))

  if refGuids[guid] then
    et.G_Print(string.format("GUID %s is already in\n", guid))
    return 1
  end

  if et.trap_Argc() > 2 then
    comment = et.ConcatArgs(2)
  end

  local fd, len = et.trap_FS_FOpenFile(conf['guidfile'], et.FS_APPEND)

  if len == 1 then
    et.G_Print(string.format("Could not open %s\n", conf['guidfile']))
    return 1
  end

  local data = string.format("%s|%s\n", guid, comment)

  et.trap_FS_Write(data, string.len(data), fd)
  et.trap_FS_FCloseFile(fd)

  refGuids[guid] = true
  et.G_Print(string.format("GUID %s successfully added\n", guid))

  return 1
end

-- vim: tabstop=2 shiftwidth=2 softtabstop=2 expandtab
